/*
* tkBiotext.c --
*
*	This module implements "Biotext" widgets that are object based. A
*	"Biotext" is a widget that displays a single Biotext that can be moved
*	around and resized. This file is intended as an example of how to
*	build a widget; it isn't included in the normal wish, but it is
*	included in "tktest".
*
* Copyright (c) 1997 Sun Microsystems, Inc.
*
* See the file "license.terms" for information on usage and redistribution of
* this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
* RCS: @(#) $Id: tkBiotext.c,v 1.13 2008/10/17 23:18:37 nijtmans Exp $
*/

#include "tkBiotext.h"

#ifndef DLLEXPORT
#define DLLEXPORT __declspec(dllexport)
#endif 

#ifndef DEBUG
#define DEBUG 1
#endif

#ifdef USE_XFT
#define WITHXFT 1
#endif

static Tk_ClassProcs BiotextClass = {
  sizeof(Tk_ClassProcs),	/* size */
  BiotextWorldChanged,		/* worldChangedProc */
};

/*
 * Information used for argv parsing.
 */

static const Tk_OptionSpec BiotextOptionSpecs[] = {
  {TK_OPTION_BORDER, "-background", "background", "Background",
   "black", -1, Tk_Offset(Biotext, bgBorder),
   0, (ClientData) "#d9d9d9", 0},
  {TK_OPTION_SYNONYM, "-bd", NULL, NULL,
   NULL, 0, 0, 0, (ClientData) "-borderwidth", 0},
  {TK_OPTION_SYNONYM, "-bg", NULL, NULL,
   NULL, 0, 0, 0, (ClientData) "-background", 0},
  {TK_OPTION_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
   "2", Tk_Offset(Biotext, borderWidthPtr), -1,
   TK_OPTION_NULL_OK, NULL, 0},
  {TK_OPTION_STRING, "-class", "class", "Class",
   "Biotext", -1, Tk_Offset(Biotext, className),
   0, NULL, 0},
  {TK_OPTION_COLOR, "-cursorcolor", NULL, NULL,
   "yellow", -1, Tk_Offset(Biotext, cursorColor),
   TK_OPTION_NULL_OK, 0, 0},
  {TK_OPTION_BOOLEAN, "-dbl", "doubleBuffer", "DoubleBuffer",
   "1", -1, Tk_Offset(Biotext, doubleBuffer),
   0, 0, 0},
  {TK_OPTION_SYNONYM, "-fg", NULL, NULL,
   NULL, 0, 0, 0, (ClientData) "-foreground", 0},
  {TK_OPTION_BORDER, "-foreground", "foreground", "Foreground",
   "white", -1, Tk_Offset(Biotext, fgBorder),
   0, (ClientData) "white", 0},
  {TK_OPTION_STRING, "-height", "height", "Height",
   "20", Tk_Offset(Biotext, heightCharPtr), -1,
   0, 0, 0},
  {TK_OPTION_INT, "-insertofftime", "insertOffTime", "OffTime",
   "300", -1, Tk_Offset(Biotext, insertOffTime),
   0, 0, 0},
  {TK_OPTION_INT, "-insertontime", "insertOnTime", "OnTime",
   "600", -1, Tk_Offset(Biotext, insertOnTime),
   0, 0, 0},
  {TK_OPTION_RELIEF, "-relief", "relief", "Relief",
   "flat", -1, Tk_Offset(Biotext, relief),
   TK_OPTION_NULL_OK, NULL, 0},
  {TK_OPTION_STRING_TABLE, "-state", "state", "State",
   "normal", -1, Tk_Offset(Biotext, state),
   0, (ClientData) stateStrings, 0},
  {TK_OPTION_STRING, "-width", "width", "Width",
   "60", Tk_Offset(Biotext, widthCharPtr), -1,
   0, 0, 0},
  {TK_OPTION_STRING, "-xscrollcommand", "xScrollCommand", "ScrollCommand",
   "", -1, Tk_Offset(Biotext, xscrollCmd),
   TK_OPTION_NULL_OK, 0, 0},
  {TK_OPTION_STRING, "-yscrollcommand", "yScrollCommand", "ScrollCommand",
   "", -1, Tk_Offset(Biotext, yscrollCmd),
   TK_OPTION_NULL_OK, 0, 0},
  {TK_OPTION_END, NULL, NULL, NULL, NULL, 0, 0, 0, 0, 0}
};

/*
 *--------------------------------------------------------------
 *
 * BiotextObjCmd --
 *
 *	This procedure is invoked to process the 
 *      "Biotext" Tcl command. It
 *	creates a new "Biotext" widget.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	A new widget is created and configured.
 *
 *--------------------------------------------------------------
 */
int
BiotextObjCmd(ClientData clientData,
	      Tcl_Interp *interp,
	      int objc,
	      Tcl_Obj *const objv[]) 
{
  register Biotext *BiotextPtr;
  Tk_Window tkwin;
  Tk_OptionTable optionTable;

  if (objc < 2) {
    Tcl_WrongNumArgs(interp, 1, objv, "pathName ?-option value ...?");
    return TCL_ERROR;
  }
    
  tkwin = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp), Tcl_GetString(objv[1]), NULL);
  if (tkwin == NULL) {
    return TCL_ERROR;
  }

  /*
   * Create the option table for this widget class.
   * If it has already been
   * created, the refcount will get bumped and j
   * ust the pointer will be
   * returned. The refcount getting bumped does
   * not concern us, because Tk
   * will ensure the table is deleted when the 
   * interpreter is destroyed.
   */
  optionTable = Tk_CreateOptionTable(interp, BiotextOptionSpecs);
  if (optionTable == NULL) 
    return TCL_ERROR;

  /*
   * Initialize the widget record. 
   * The memset allows us to set
   * just the non-NULL/0 items.
   */
  BiotextPtr = (Biotext *) ckalloc(sizeof(Biotext));
  memset(BiotextPtr, 0, sizeof(Biotext));
  
  /* ===========================================
   * Initialize element of the Biotext structure
   */
  BiotextPtr->interp         = interp;
  BiotextPtr->tkwin          = tkwin;
  BiotextPtr->display        = Tk_Display(tkwin);
  BiotextPtr->screen         = Tk_Screen(tkwin);
  BiotextPtr->gc             = None;
  BiotextPtr->optionTable    = optionTable;
  /* widget config */
  BiotextPtr->state          = STATE_NORMAL;
  BiotextPtr->doubleBuffer   = 1;
  BiotextPtr->reliefPtr      = NULL;
  BiotextPtr->relief         = TK_RELIEF_FLAT;
  BiotextPtr->borderWidth    = 0;
  BiotextPtr->borderWidthPtr = NULL;
  BiotextPtr->inset          = 1;
  BiotextPtr->widgetCmd      = Tcl_CreateObjCommand(interp, Tk_PathName(tkwin), BiotextWidgetObjCmd, BiotextPtr, (Tcl_CmdDeleteProc *)BiotextDeletedProc);
  /* widget dimensions */
  BiotextPtr->width          = -1;
  BiotextPtr->height         = -1;
  BiotextPtr->widthChar      = 0;
  BiotextPtr->widthCharPtr   = NULL;
  BiotextPtr->heightChar     = 0;
  BiotextPtr->heightCharPtr  = NULL;
  /* widget content */
  BiotextPtr->leftIndex      = 0;
  BiotextPtr->topIndex       = 0;
  BiotextPtr->SeqMat         = NULL;
  BiotextPtr->nbSeqs         = -1;
  /* mapping */
  BiotextPtr->MapSeq           = NULL;
  BiotextPtr->Mapping         = NULL;
  BiotextPtr->nbMapping       = 0;
  /* tagging */
  BiotextPtr->TagSeq          = NULL;
  BiotextPtr->Tags           = NULL;
  BiotextPtr->nbTags         = 0;
  BiotextPtr->Lock           = 1;
  /* Set font to be used, get font characteristics */
  BiotextPtr->xft            = 1;
  BiotextPtr->FontName        = NULL;
  BiotextPtr->FontXft         = NULL;
  strcpy(BiotextPtr->FontFamily,"Courier");
  BiotextPtr->FontSize        = 8;
  strcpy(BiotextPtr->FontWeight,"normal");
  BiotextPtr->charWidth       = 1;
  BiotextPtr->charHeight      = 1;
  /* Group initialisation */
  BiotextPtr->nbrGrps         = 0;
  BiotextPtr->GroupList       = NULL;
  /* Cut and Paste initialisation */
  BiotextPtr->nbrSeqsCache    = 0;
  BiotextPtr->SeqGrpCache     = NULL;
  BiotextPtr->SeqMatCache     = NULL;
  BiotextPtr->MapSeqCache       = NULL;
  BiotextPtr->TagSeqCache       = NULL;

  Tcl_Preserve(BiotextPtr->tkwin);

  Tk_SetClass(tkwin, "Biotext");
  Tk_SetClassProcs(BiotextPtr->tkwin, &BiotextClass, BiotextPtr);

  /*
   * set up event proc
   */
  Tk_CreateEventHandler(BiotextPtr->tkwin,
			ExposureMask|StructureNotifyMask|FocusChangeMask,
			BiotextObjEventProc, 
			BiotextPtr);

  if (Tk_InitOptions(interp, (char *) BiotextPtr, optionTable, tkwin) != TCL_OK) {
    Tk_DestroyWindow(BiotextPtr->tkwin);
    
    return TCL_ERROR;
  }

  BiotextSetFont(BiotextPtr);
  if (BiotextConfigure(interp, BiotextPtr, objc-2, objv+2) != TCL_OK) {
    Tk_DestroyWindow(BiotextPtr->tkwin);
    
    return TCL_ERROR;
  }

  Tcl_ResetResult(interp);
  Tcl_SetObjResult(interp, Tcl_NewStringObj((char *)Tk_PathName(tkwin), -1));

  return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * BiotextWidgetObjCmd --
 *
 *	This procedure is invoked to process the Tcl command that corresponds
 *	to a widget managed by this module. See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
BiotextWidgetObjCmd(ClientData clientData,
		    Tcl_Interp *interp,
		    int objc,
		    Tcl_Obj * const objv[]) 
{
  register Biotext *BiotextPtr = (Biotext *)clientData;
  int index, result = TCL_OK;
  Tcl_Obj *resultObjPtr;

  static CONST84 char *BiotextOptions[] = {"addseqs", "cget", "chars", "clean", "configure", "copy", "cursor", "cut", "delchars", "delete", "font", "group",  "index", "insert", "lock", "map", "mapping", "tag", "output", "paste", "push", "see", "sequences", "ungroup", "xview", "yview", NULL};
  enum selOptions {
    Biotext_ADDSEQS, Biotext_CGET, Biotext_CHARS, Biotext_CLEAN, Biotext_CONFIGURE, Biotext_COPY, Biotext_CURSOR, Biotext_CUT, Biotext_DELCHARS, Biotext_DELETE, Biotext_FONT, Biotext_GROUP, Biotext_INDEX, Biotext_INSERT, Biotext_LOCK, Biotext_MAP, Biotext_MAPPING, Biotext_TAG, Biotext_OUTPUT, Biotext_PASTE, Biotext_PUSH, Biotext_SEE, Biotext_SEQUENCES, Biotext_UNGROUP, Biotext_XVIEW, Biotext_YVIEW,
  };

  if (objc < 2) {
    Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg...?");
    return TCL_ERROR;
  }

  if (Tcl_GetIndexFromObj(interp, objv[1], BiotextOptions, "command", 0, &index) != TCL_OK) {
    return TCL_ERROR;
  }

  Tcl_Preserve((ClientData) BiotextPtr);

  switch ((enum selOptions) index) {
  case Biotext_ADDSEQS:
    if (BiotextAddSeqsCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_CGET:
    if (objc != 3) {
      Tcl_WrongNumArgs(interp, 2, objv, "option");
      goto error;
    }
    resultObjPtr = Tk_GetOptionValue(interp, (char *) BiotextPtr, BiotextPtr->optionTable, objv[2], BiotextPtr->tkwin);
    if (resultObjPtr == NULL) {
      result = TCL_ERROR;
    } else {
      Tcl_SetObjResult(interp, resultObjPtr);
    }
    break;
  case Biotext_CLEAN:
    BiotextCleanCmd(BiotextPtr, interp, objc, objv);
    result = TCL_OK;

    break;
  case Biotext_CONFIGURE:
    resultObjPtr = NULL;
    if (objc <= 3) {
      resultObjPtr = Tk_GetOptionInfo(interp, (char *) BiotextPtr,BiotextPtr->optionTable, (objc == 3) ? objv[2] : NULL, BiotextPtr->tkwin);
      if (resultObjPtr == NULL) {
	Tcl_Release(BiotextPtr);
	return TCL_ERROR;
      }

      Tcl_SetObjResult(interp, resultObjPtr);
    } else {
      /* set the configuration options */
      result = BiotextConfigure(interp, BiotextPtr, objc-2, objv+2);
    }

    break;
    /* 
       case Biotext_SELECTION:
       if (objc < 3) {
       Tcl_WrongNumArgs(interp, 3, objv, "option arg ?arg?");
       goto error;
       }
       BiotextSelectionCmd(BiotextPtr, interp, objc, objv);
       break;
    */
  case Biotext_SEQUENCES:
    BiotextSequencesCmd(BiotextPtr, interp, objc, objv);
    break;
  case Biotext_INSERT:
    if (BiotextInsertCmd(BiotextPtr, interp, objc, objv) == TCL_ERROR) goto error;
    break;
  case Biotext_DELETE:
    if (BiotextDeleteCmd(BiotextPtr, interp, objc, objv) == TCL_ERROR) goto error;
    break;
  case Biotext_MAPPING:
    BiotextMappingCmd(BiotextPtr, interp, objc, objv);
    break;
  case Biotext_TAG:
    BiotextTagCmd(BiotextPtr, interp, objc, objv);
    break;
  case Biotext_MAP:
    BiotextMapCmd(BiotextPtr, interp, objc,objv);
    break;
  case Biotext_XVIEW:
    if (BiotextXViewCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_YVIEW:
    if (BiotextYViewCmd(BiotextPtr, interp, objc, objv)) goto error;
    break;
  case Biotext_GROUP:
    if (BiotextGroupCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_UNGROUP:
    if (BiotextUngroupCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_SEE:
    if (BiotextSeeCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_INDEX:
    if (BiotextIndexCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_PUSH:
    if (BiotextPushCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_CURSOR:
    if (BiotextCursorCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_CUT:
    if (BiotextCutCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_COPY:
    if (BiotextCopyCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_PASTE:
    if (BiotextPasteCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_FONT:
    if (BiotextFontCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_OUTPUT:
    if (BiotextOutputCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_DELCHARS:
    if (BiotextDelcharsCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_LOCK:
    if (BiotextLockCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_CHARS:
    if (BiotextCharsCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
  }
    
  Tcl_Release(BiotextPtr);
  return result;

 error:
  Tcl_Release(BiotextPtr);
  return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextConfigCursor --
 *	Configures the timer depending on the 
 *      state of the Biotext widget.
 *	Equivalent to BiotextFocusProc.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The cursor will be switched off/on.
 *
 *----------------------------------------------------------------------
 */
void
BiotextConfigCursor(register Biotext *BiotextPtr)
{
  if ((BiotextPtr->state == STATE_DISABLED) || (BiotextPtr->tkwin == NULL) || (! (Tk_IsMapped(BiotextPtr->tkwin)))) 
    return;
  
  /* make sure nothing existed */
  Tcl_DeleteTimerHandler(BiotextPtr->cursorTimer);
  /*
   * To have a cursor, we have to have focus and 
   * allow edits
   */
  if (BiotextPtr->flags & GOT_FOCUS) {
    /*
     * Turn the cursor ON
     */
    if (! (BiotextPtr->flags & CURSOR_ON)) {
      BiotextPtr->flags |= CURSOR_ON;
    }
    
    /* set up the first timer */
    if (BiotextPtr->insertOffTime != 0) {
      BiotextPtr->cursorTimer =	Tcl_CreateTimerHandler(BiotextPtr->insertOnTime, BiotextCursorEvent, (ClientData) BiotextPtr);
    }
  } else {
    /* No focus
     * Turn the cursor OFF
     */
    if ((BiotextPtr->flags & CURSOR_ON)) {
      BiotextPtr->flags &= ~CURSOR_ON;
    }
    
    /* and disable the timer */
      BiotextPtr->cursorTimer = NULL;
  }  

  return;
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextConfigure --
 *
 *	This procedure is called to process an 
 *      argv/argc list in conjunction
 *	with the Tk option database to configure (or 
 *      reconfigure) a Biotext
 *	widget.
 *
 * Results:
 *	The return value is a standard Tcl result. 
 *      If TCL_ERROR is returned, then the 
 *      interp's result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as 
 *      colors, border width, etc. get set
 *	for BiotextPtr; old resources get freed, 
 *      if there were any.
 *
 *----------------------------------------------------------------------
 */
int
BiotextConfigure(Tcl_Interp *interp, 
		 register Biotext *BiotextPtr,
		 int objc,
		 Tcl_Obj *const objv[])	
{
  int error;
  Tk_SavedOptions savedOptions;
  Tk_3DBorder bgBorder;
  Tcl_Obj *errorResult = NULL;
  
  for (error = 0; error <= 1; error++) {
    if (! error) {
      /*
       * First pass: set options to new values.
       */
      
      if (Tk_SetOptions(interp, (char *) BiotextPtr,
			BiotextPtr->optionTable, objc, objv,
			BiotextPtr->tkwin, &savedOptions, NULL) != TCL_OK) {
	continue;
      }
    } else {
      /*
       * Second pass: restore options to old values.
       */
      
      errorResult = Tcl_GetObjResult(interp);
      Tcl_IncrRefCount(errorResult);
      Tk_RestoreSavedOptions(&savedOptions);
    }
    
    /*
     * Set the background for the window and create
     * a graphics context for use during redisplay.
     */
    bgBorder = BiotextPtr->bgBorder;
    Tk_SetBackgroundFromBorder(BiotextPtr->tkwin, bgBorder);

    /*
     * Restart the cursor timing sequence in case the 
     * on-time or off-time just changed. 
     */
    if (BiotextPtr->flags & GOT_FOCUS) 
      BiotextConfigCursor(BiotextPtr);
    
    break;
  }

  if (! error) {
    Tk_FreeSavedOptions(&savedOptions);
  }

  /* in case background/foreground changed */
  Biotext_AssignColors(BiotextPtr,1);
  
  BiotextWorldChanged(BiotextPtr);
  if (error) {
    Tcl_SetObjResult(interp, errorResult);
    Tcl_DecrRefCount(errorResult);
    return TCL_ERROR;
  } else {
    return TCL_OK;
  }
}

/*
 *--------------------------------------------------------------
 *
 * BiotextObjEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for 
 *      various events on Biotexts.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get 
 *      cleaned up. When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

void
BiotextObjEventProc(ClientData clientData,	/* Information about window. */
		    XEvent *eventPtr)		/* Information about event. */
{
  register Biotext *BiotextPtr = clientData;
  
  switch (eventPtr->type) {
  case Expose:
    BiotextPtr->flags |= UPDATE_SCROLLBAR;
    BiotextEventuallyRedraw(BiotextPtr);
    
    break;
  case DestroyNotify:
    if (! (BiotextPtr->flags & BIOTEXT_DELETED)) {
      BiotextPtr->flags |= (BIOTEXT_DELETED);
    }
    if (BiotextPtr->tkwin != NULL) {
      if (BiotextPtr->gc != None) {
	Tk_FreeGC(BiotextPtr->display, BiotextPtr->gc);
      }
      Tk_FreeConfigOptions((char *) BiotextPtr, BiotextPtr->optionTable,BiotextPtr->tkwin);
      BiotextPtr->tkwin = NULL;
      Tcl_DeleteCommandFromToken(BiotextPtr->interp,BiotextPtr->widgetCmd);
    }

    break;
  case ConfigureNotify:
    /*
     * Recalculate the number of lines/columns
     * Then arrange for the window to be redisplayed.
     */    
    if (! (Tk_IsMapped(BiotextPtr->tkwin))) 
      return;
    
    int bdi, width, height, nC, nR;
    bdi = 2*(BiotextPtr->borderWidth+BiotextPtr->inset);
    width  = Tk_Width(BiotextPtr->tkwin) - bdi;
    height = Tk_Height(BiotextPtr->tkwin) - bdi;
    nC = width / BiotextPtr->charWidth;
    nR = height / BiotextPtr->charHeight;
    BiotextPtr->widthChar  = nC;
    BiotextPtr->heightChar = nR;

    BiotextPtr->flags |= UPDATE_SCROLLBAR;
    BiotextEventuallyRedraw(BiotextPtr);
    
    break;
  case FocusIn:
  case FocusOut:
    if (eventPtr->type == FocusOut) {
      BiotextPtr->flags &= ~GOT_FOCUS;
    } else {
      BiotextPtr->flags |= GOT_FOCUS;
    }
    /* cancel/start the timer */
    BiotextConfigCursor(BiotextPtr);
    BiotextEventuallyRedraw(BiotextPtr);
    
    break;
  }
  
  return;
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted. If the
 *	widget isn't already in the process of being destroyed, this command
 *	destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */

void
BiotextDeletedProc(
		   ClientData clientData)	/* Pointer to widget record for widget. */
{
  register Biotext *BiotextPtr = clientData;
  Tk_Window tkwin = BiotextPtr->tkwin;

  /*
   * This procedure could be invoked either because the window was destroyed
   * and the command was then deleted (in which case tkwin is NULL) or
   * because the command was deleted, and then this procedure destroys the
   * widget.
   */

  if (tkwin != NULL) {
    Tk_DestroyWindow(tkwin);
  }
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextEventuallyRedraw --
 *
 *	Ensure that a biotext is eventually redrawn 
 *      on the display.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets redisplayed. Right now we 
 *      don't do selective redisplays: the whole 
 *      window will be redrawn.
 *      This doesn't seem to hurt performance 
 *      noticeably, but if it does then this could be
 *	changed.
 *
 *----------------------------------------------------------------------
 */
void
BiotextEventuallyRedraw(register Biotext *BiotextPtr) 
{
  if ((BiotextPtr->flags & BIOTEXT_DELETED) || (BiotextPtr->tkwin == NULL) || ! Tk_IsMapped(BiotextPtr->tkwin)) 
    return;

  /*
   * Right now we don't do selective redisplays: 
   * the whole window will be redrawn. This doesn't 
   * seem to hurt performance noticeably, but if it
   * does then this could be changed.
   */

  if (! (BiotextPtr->flags & REDRAW_PENDING)) {
    BiotextPtr->flags |= REDRAW_PENDING;
    Tcl_DoWhenIdle(BiotextDisplay, (ClientData) BiotextPtr);
  }

  return;
}

/*
 *--------------------------------------------------------------
 *
 * BiotextDisplay --
 *
 *	This procedure redraws the contents of a Biotext window.
 *      It is invoked as a do-when-idle handler, so it only runs
 *      when there's nothing else for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */
void
BiotextDisplay(ClientData clientData) 
{
  register Biotext *BiotextPtr = (Biotext *) clientData;
  Tk_Window tkwin = BiotextPtr->tkwin;
  Display *display = BiotextPtr->display;
  Pixmap pm = None;
  Drawable d;

  /*
   * reset the pending flag
   */
  BiotextPtr->flags &= ~REDRAW_PENDING;
  if ((BiotextPtr->flags & BIOTEXT_DELETED) || ! Tk_IsMapped(tkwin)) 
    return;

  /*
   * Update the scrollbar if that's needed.
   */
  if (BiotextPtr->flags & UPDATE_SCROLLBAR) {
    BiotextPtr->flags &= ~UPDATE_SCROLLBAR;
    
    /*
     * Preserve/Release because updating the 
     * scrollbar can have the side-effect of 
     * destroying or unmapping the biotext widget.
     */
    Tcl_Preserve(BiotextPtr);
    BiotextUpdateXScrollbar(BiotextPtr);
    BiotextUpdateYScrollbar(BiotextPtr);
    Tcl_Release(BiotextPtr);
  }
  
  /*
   * Create a pixmap for doublebuffering if necessary.
   */
  if (BiotextPtr->doubleBuffer) {
    pm = Tk_GetPixmap(display, Tk_WindowId(tkwin), Tk_Width(tkwin), Tk_Height(tkwin), DefaultDepthOfScreen(Tk_Screen(tkwin)));
    d = pm;
  } else {
    d = Tk_WindowId(tkwin);
  }
  
  /*
   * Redraw the widget's background and border.
   */
  Tk_Fill3DRectangle(tkwin, d, BiotextPtr->bgBorder, 0, 0, Tk_Width(tkwin), Tk_Height(tkwin), 0, TK_RELIEF_FLAT);

  /*
   * Display the Biotext.
   */

  GC newgc;
  XGCValues gcValues;
  gcValues.graphics_exposures = False;
  gcValues.font = Tk_FontId(BiotextPtr->tkfont);
  gcValues.foreground = Tk_3DBorderColor(BiotextPtr->fgBorder)->pixel;
  newgc = Tk_GetGC(tkwin, GCGraphicsExposures|GCFont|GCForeground, &gcValues);
  if (BiotextPtr->gc != NULL) 
    Tk_FreeGC(BiotextPtr->display,BiotextPtr->gc);
  BiotextPtr->gc = newgc;

  /*
   * If Empty widget, then return.
   */
  if (BiotextPtr->SeqMat == NULL) {
    XCopyArea(display, pm, Tk_WindowId(tkwin), BiotextPtr->gc, 0, 0, (unsigned) Tk_Width(tkwin), (unsigned) Tk_Height(tkwin), 0, 0);
    Tk_FreePixmap(display, pm);
    
    return;
  }

  /*
   * Draw the character matrix according to the 
   * current mapping.
   */
  BiotextDrawAlignment(BiotextPtr, d);

  /*
   * Draw the blinking block cursor if any
   */
  if ((BiotextPtr->state == STATE_NORMAL) && (BiotextPtr->flags & GOT_FOCUS) && (BiotextPtr->flags & CURSOR_ON)) {
    int xP , yP, wc, hc, col, row;

    col = BiotextPtr->currentC;
    row = BiotextPtr->currentR;
    if (Biotext_IsVisible(BiotextPtr, col, row, 1)) {
      wc  = BiotextPtr->charWidth;
      hc  = BiotextPtr->charHeight;
      Biotext_Coords2Pixels(BiotextPtr, col, row, &xP, &yP, 1);

      Tk_Fill3DRectangle(tkwin, d, Tk_Get3DBorder(BiotextPtr->interp, tkwin, Tk_NameOfColor(BiotextPtr->cursorColor)), xP, yP, wc, hc, 1, TK_RELIEF_RAISED);
    }
  }

  /*
   * If double-buffered, copy to the screen and 
   * release the pixmap.
   */
  if (BiotextPtr->doubleBuffer) {
    //XGCValues gcValues;
    //gcValues.foreground = Tk_3DBorderColor(BiotextPtr->fgBorder)->pixel;
    //GC newgc = BiotextPtr->gc;
    //XChangeGC(display, newgc, GCForeground, &gcValues);
    XCopyArea(display, pm, Tk_WindowId(tkwin), BiotextPtr->gc, 0, 0, (unsigned) Tk_Width(tkwin), (unsigned) Tk_Height(tkwin), 0, 0);
    Tk_FreePixmap(display, pm);
  }

  return;
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextDestroy --
 *
 *	This procedure is invoked by 
 *      Tcl_EventuallyFree or Tcl_Release to
 *	clean up the internal structure of a Biotext 
 *      at a safe time when no-one is using it anymore
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the Biotext is 
 *      freed up.
 *
 *----------------------------------------------------------------------
 */

void
BiotextDestroy(char *memPtr) /* Info about Biotext widget. */
{
  register Biotext *BiotextPtr = (Biotext *) memPtr;

  ckfree((char *) BiotextPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextVisibleXRange --
 *
 *	Return information about the range of the 
 *      Biotext in X-coord that is currently
 *	visible.
 *
 * Results:
 *	*firstPtr and *lastPtr are modified to hold 
 *      fractions between 0 and 1
 *	identifying the range of characters visible 
 *      in the Biotext.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
BiotextVisibleXRange(register Biotext *BiotextPtr,
		     double *firstPtr,  
		     double *lastPtr) 
{
  int dif;
  
  if (BiotextPtr->lgSeqs == 0) {
    *firstPtr = 0.0;
    *lastPtr = 1.0;
  } else {
    dif = BiotextPtr->lgSeqs - (BiotextPtr->leftIndex + BiotextPtr->widthChar);
    if (BiotextPtr->lgSeqs <= BiotextPtr->widthChar) {
      BiotextPtr->leftIndex = 0;
    } else if (dif < 0) {
      BiotextPtr->leftIndex += dif;
    }

    *firstPtr = (double) BiotextPtr->leftIndex / BiotextPtr->lgSeqs;
    *lastPtr = (double) (BiotextPtr->leftIndex + BiotextPtr->widthChar) / BiotextPtr->lgSeqs;
  }
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextVisibleYRange --
 *
 *	Return information about the range of the 
 *      Biotext in Y-coord that is currently
 *	visible.
 *
 * Results:
 *	*firstPtr and *lastPtr are modified to hold 
 *      fractions between 0 and 1
 *	identifying the range of characters visible 
 *      in the Biotext.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
BiotextVisibleYRange(register Biotext *BiotextPtr,
		     double *firstPtr,
		     double *lastPtr) 
{
  int dif;
  
  if (BiotextPtr->nbSeqs <= 0) {
    *firstPtr = 0.0;
    *lastPtr = 1.0;
  } else {
    dif = BiotextPtr->nbSeqs - (BiotextPtr->topIndex + BiotextPtr->heightChar);
    if (BiotextPtr->nbSeqs<= BiotextPtr->heightChar) {
      BiotextPtr->topIndex = 0;
    } else if (dif < 0) {
      BiotextPtr->topIndex += dif;
    }

    *firstPtr = (double) BiotextPtr->topIndex / BiotextPtr->nbSeqs;
    *lastPtr = (double) (BiotextPtr->topIndex + BiotextPtr->heightChar) / (double) BiotextPtr->nbSeqs;
  }
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextUpdateXScrollbar --
 *
 *	This function is invoked whenever 
 *      information has changed in an Biotext
 *	in a way that would invalidate a X -xscrollbar
 *      display. If there is an
 *	associated X -scrollbar, then this function 
 *      updates it by invoking a Tcl
 *	command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A Tcl command is invoked, and an additional 
 *      command may be
 *	invoked to process errors in the command.
 *
 *----------------------------------------------------------------------
 */

void
BiotextUpdateXScrollbar(Biotext *BiotextPtr) 
{
  char firstStr[TCL_DOUBLE_SPACE], lastStr[TCL_DOUBLE_SPACE];
  int code;
  double first, last;
  Tcl_Interp *interp;

  if (BiotextPtr->xscrollCmd == NULL) {
    return;
  }

  interp = BiotextPtr->interp;
  Tcl_Preserve(interp);

  BiotextVisibleXRange(BiotextPtr, &first, &last);
  Tcl_PrintDouble(NULL, first, firstStr);
  Tcl_PrintDouble(NULL, last, lastStr);
  code = Tcl_VarEval(interp, BiotextPtr->xscrollCmd, " ", firstStr, " ", lastStr, NULL);
  if (code != TCL_OK) {
    Tcl_AddErrorInfo(interp, "\n    (horizontal scrolling command executed by ");
    Tcl_AddErrorInfo(interp, Tk_PathName(BiotextPtr->tkwin));
    Tcl_AddErrorInfo(interp, ")");
    Tcl_BackgroundError(interp);
  }

  Tcl_ResetResult(interp);
  Tcl_Release(interp);

  return;
}


/*
 *----------------------------------------------------------------------
 *
 * BiotextUpdateYScrollbar --
 *
 *	This function is invoked whenever 
 *      information has changed in an Biotext
 *	in a way that would invalidate an
 *      Y -yxscrollbar display. If there is an
 *	associated X -scrollbar, then this function 
 *      updates it by invoking a Tcl
 *	command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A Tcl command is invoked, and an additional command may be
 *	invoked to process errors in the command.
 *
 *----------------------------------------------------------------------
 */

void
BiotextUpdateYScrollbar(register Biotext *BiotextPtr) 
{
  char firstStr[TCL_DOUBLE_SPACE], lastStr[TCL_DOUBLE_SPACE];
  int code;
  double first, last;
  Tcl_Interp *interp;
  
  if (BiotextPtr->yscrollCmd == NULL) {
    return;
  }
  
  interp = BiotextPtr->interp;
  Tcl_Preserve(interp);
  
  BiotextVisibleYRange(BiotextPtr, &first, &last);
  Tcl_PrintDouble(NULL, first, firstStr);
  Tcl_PrintDouble(NULL, last, lastStr);
  code = Tcl_VarEval(interp, BiotextPtr->yscrollCmd, " ", firstStr, " ", lastStr, NULL);
  if (code != TCL_OK) {
    Tcl_AddErrorInfo(interp, "\n    (horizontal scrolling command executed by ");
    Tcl_AddErrorInfo(interp, Tk_PathName(BiotextPtr->tkwin));
    Tcl_AddErrorInfo(interp, ")");
    Tcl_BackgroundError(interp);
  }
  
  Tcl_ResetResult(interp);
  Tcl_Release(interp);
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextCursorEvent --
 *	Toggle the cursor status. 
 *      Equivalent to EntryBlinkProc.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The cursor will be switched off/on.
 *
 *----------------------------------------------------------------------
 */
void
BiotextCursorEvent(ClientData clientData)
{
  register Biotext *BiotextPtr = (Biotext *) clientData;

  if (! (BiotextPtr->flags & GOT_FOCUS) || (BiotextPtr->insertOffTime == 0) || (BiotextPtr->state != STATE_NORMAL)) {
    return;
  }
  
  BiotextPtr->cursorTimer = Tcl_CreateTimerHandler((BiotextPtr->flags & CURSOR_ON) ? BiotextPtr->insertOffTime : BiotextPtr->insertOnTime, BiotextCursorEvent, (ClientData) BiotextPtr);

  /* Toggle the cursor */
  BiotextPtr->flags ^= CURSOR_ON;

  /* Redraw */
  BiotextEventuallyRedraw(BiotextPtr);

  return;
}


/*
 * Package initialisation
 */
int DLLEXPORT Biotext_Init(Tcl_Interp *interp) 
{
  if (Tcl_InitStubs(interp, "8.5" , 0) == NULL) {
    return TCL_ERROR;
  }
  if (Tk_InitStubs(interp, "8.5" , 0) == NULL) {
    return TCL_ERROR;
  }
  
  Tcl_CreateObjCommand(interp, "biotext", BiotextObjCmd, (ClientData) NULL, (Tcl_CmdDeleteProc *)NULL);

  Tcl_PkgProvide(interp,"biotext","0.1");

  return TCL_OK;
}

/*
 *---------------------------------------------------------------------------
 *
 * BiotextWorldChanged --
 *
 *	This function is called when the world has changed in 
 *      some way and the widget needs to recompute all its 
 *      graphics contexts and determine its new geometry.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Biotext will be relayed out and redisplayed.
 *
 *---------------------------------------------------------------------------
 */

void
BiotextWorldChanged(ClientData instanceData) 
{
  XGCValues gcValues;
  GC gc = None;
  Biotext *BiotextPtr = (Biotext *) instanceData;
  
  Tk_SetBackgroundFromBorder(BiotextPtr->tkwin, BiotextPtr->bgBorder);

  if (BiotextPtr->doubleBuffer) {
    /*
     * Set up the graphical context:
     * - defaultgc is the one used when no mapping 
     */
    gcValues.graphics_exposures = False;
    gcValues.font = Tk_FontId(BiotextPtr->tkfont);
    gcValues.foreground = Tk_3DBorderColor(BiotextPtr->fgBorder)->pixel;
    gc = Tk_GetGC(BiotextPtr->tkwin,GCGraphicsExposures|GCFont|GCForeground, &gcValues);
    if (BiotextPtr->gc != NULL) 
      Tk_FreeGC(BiotextPtr->display, BiotextPtr->gc);
    BiotextPtr->gc = gc;
  }

  /*
   * Recompute the window's geometry and arrange for it to be redisplayed.
   */
  BiotextComputeGeometry(BiotextPtr);
  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);
  
  return;
}
  

/* 
valgrind -v --leak-check=yes --num-callers=10 --leak-resolution=high --show-reachable=no /home/moumou/tcl8.6b1/lubin/bin/tclsh8.6 tbiotext.tcl >& log
*/




  
