#include "tkBiotext.h"

#ifndef DEBUG
#define DEBUG 0
#endif



void Tmp_Color(Biotext *BiotextPtr, Drawable d) 
{
  int i;
  XColor cf, cb;
  XGCValues gcValues;
  Colormap cmap = Tk_GetColormap(BiotextPtr->interp, BiotextPtr->tkwin, "new");

  gcValues.graphics_exposures = False;
  gcValues.font = Tk_FontId(BiotextPtr->tkfont);
  gcValues.function = GXcopy;
  for (i=0;i<BiotextPtr->nbMapping;i++) {
    if (BiotextPtr->xft) {
#ifndef WIN32
      cf.pixel = BiotextPtr->TagMap[i].fg;
      cb.pixel = BiotextPtr->TagMap[i].bg;
      XQueryColor(BiotextPtr->display, cmap, &cf);
      XQueryColor(BiotextPtr->display, cmap, &cb);
      
#ifdef USE_XFT
      BiotextPtr->TagMap[i].Cfg.pixel = cf.pixel;
      BiotextPtr->TagMap[i].Cfg.color.red = cf.red;
      BiotextPtr->TagMap[i].Cfg.color.green = cf.green;
      BiotextPtr->TagMap[i].Cfg.color.blue = cf.blue;
      BiotextPtr->TagMap[i].Cfg.color.alpha = 0xffff;
      
      BiotextPtr->TagMap[i].Cbg.pixel = cb.pixel;
      BiotextPtr->TagMap[i].Cbg.color.red = cb.red;
      BiotextPtr->TagMap[i].Cbg.color.green = cb.green;
      BiotextPtr->TagMap[i].Cbg.color.blue = cb.blue;
      BiotextPtr->TagMap[i].Cbg.color.alpha = 0xffff;
#endif
#endif
    } else {
      gcValues.background = BiotextPtr->TagMap[i].bg;
      gcValues.foreground = BiotextPtr->TagMap[i].fg;
      
      BiotextPtr->TagMap[i].gc = XCreateGC(BiotextPtr->display, d, GCFunction|GCGraphicsExposures|GCFont|GCForeground|GCBackground, &gcValues);
    }
  }

  return;
}


/*
 *--------------------------------------------------------------
 *
 * BiotextGetIndex --
 *	Parse an index into a biotext and return 
 *      either its value or an error.
 *
 * Results:
 *	A standard Tcl result.  If all went well, 
 *      then *row,*col is
 *	filled in with the index corresponding to 
 *      string.  If an
 *	error occurs then an error message is 
 *      left in interp result.
 *
 * Side effects:
 *      None.      
 *
 *--------------------------------------------------------------
 */
int
BiotextGetIndex(register Biotext *BiotextPtr, 
		char *str, 
		int *row_p, 
		int *col_p) 
{
  int r, c, len = strlen(str);
  char dummy;
  
  if (str[0] == '@') {	/* @x,y coordinate */ 
    int x, y;
    
    /* Make sure it won't work for "2,3extrastuff" */
    if (sscanf(str+1, "%d,%d%c", &x, &y, &dummy) != 2) {
      goto IndexError;
    }
    Biotext_Pixels2Coords(BiotextPtr,x,y,&c,&r);
  } else if (*str == '-' || isdigit(str[0])) {
    /* Make sure it won't work for "2,3extrastuff" */
    if (sscanf(str, "%d,%d%c", &r, &c, &dummy) != 2) {
      goto IndexError;
    }    
  } else if (strncmp(str, "end", len) == 0) {		/* end */
    r = BiotextPtr->nbSeqs-1;
    c = BiotextPtr->lgSeqs-1;
  } else if (strncmp(str, "current", len) == 0) {
    r = BiotextPtr->currentR;
    c = BiotextPtr->currentC;
  } else {
  IndexError:
    Tcl_AppendStringsToObj(Tcl_GetObjResult(BiotextPtr->interp), "bad index \"", str, "\": must be end, @x,y, or <row>,<col>", (char *)NULL);
    return TCL_ERROR;
  }
  
  /* check for dummies values */
  if (r <  0) r = 0;
  if (r >= BiotextPtr->nbSeqs - 1) r=BiotextPtr->nbSeqs-1;
  if (c <  0) c = 0;
  if (c >= BiotextPtr->lgSeqs - 1) c=BiotextPtr->lgSeqs-1;
  
  if (row_p) *row_p = r;
  if (col_p) *col_p = c;
  
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Biotext_Pixels2Coords --
 *      Gives the correspondance from pixels 
 *      coordinates to characters coordinates
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.      
 *
 *--------------------------------------------------------------
 */

void Biotext_Pixels2Coords(Biotext *BiotextPtr,
			   int x,
			   int y,
			   int *c,
			   int *r) 
{

  *c = (int)((double)(x - (BiotextPtr->borderWidth + BiotextPtr->inset))/BiotextPtr->charWidth) + BiotextPtr->leftIndex;
  *r = (int)((double)(y - (BiotextPtr->borderWidth + BiotextPtr->inset))/BiotextPtr->charHeight) + BiotextPtr->topIndex;
}

/*
 *--------------------------------------------------------------
 *
 * Biotext_Coords2Pixels --
 *      Gives the correspondance from character 
 *      coordintates to pixels coordinates.
 *      The pixels coordinates are the coordinates of
 *      the upper-left corner of the character.
*
 * Results:
 *      None.
 *
 * Side effects:
 *      None.      
 *
 *--------------------------------------------------------------
 */

void Biotext_Coords2Pixels(Biotext *BiotextPtr,
			   int c,
			   int r,
			   int *x,
			   int *y,
			   int flag) 
{
  /*
   * flag = 0 : the output values are on a absolute 
   * scale.
   * flag = 1 : the output values are relative to the
   * border of the window
   */

  if (flag) {
    *x = (c - BiotextPtr->leftIndex)*BiotextPtr->charWidth + BiotextPtr->borderWidth + BiotextPtr->inset;
    *y = (r - BiotextPtr->topIndex )*BiotextPtr->charHeight + BiotextPtr->borderWidth + BiotextPtr->inset;
  } else {
    *x = c*BiotextPtr->charWidth + BiotextPtr->borderWidth + BiotextPtr->inset;
    *y = r*BiotextPtr->charHeight + BiotextPtr->borderWidth + BiotextPtr->inset;
  }
}

/*
 *--------------------------------------------------------------
 *
 * Biotext_IsVisible --
 *      Returns 1 if pixel/character at x,y
 *      is visible on the screen. Pixel/Char depends
 *      on the flag value.
 * 
 * Results:
 *      None.
 *
 * Side effects:
 *      None.      
 *
 *--------------------------------------------------------------
 */
int Biotext_IsVisible(register Biotext *BiotextPtr, 
		      int xc, 
		      int yc, 
		      int flag) 
{
  /* flag = 1 -> character, 
     flag = 0 -> pixels
  */

  if (flag) {
    if ((xc >= BiotextPtr->leftIndex) && (yc >= BiotextPtr->topIndex) && (xc <= BiotextPtr->leftIndex + BiotextPtr->widthChar) && (yc <= BiotextPtr->topIndex + BiotextPtr->heightChar)) {
      return (1);
    } else {
      return (0);
    }
  } else {
    if ( (xc >= 0 && yc >= 0) && (xc <= BiotextPtr->width && yc <= BiotextPtr->height)) {
      return (1);
    } else {
      return (0);
    }
  }
}


/*--------------------------------------------------------------
 *
 * BiotextDeleteGroup --
 *
 *      This routine deletes a group
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextDeleteGroup(register Biotext *BiotextPtr,
		   int nm) 
{
  int i, val;
  Tcl_Obj *el;

  for(i=0;i<BiotextPtr->nbrGrps;i++) {
    Tcl_ListObjIndex(BiotextPtr->interp, BiotextPtr->GroupList, i, &el);
    Tcl_GetIntFromObj(BiotextPtr->interp, el, &val);
    if (val == nm) {
      Tcl_ListObjReplace(BiotextPtr->interp, BiotextPtr->GroupList, i, 1, 0, NULL);
      break;
    }
  }
  
  BiotextPtr->nbrGrps--;
  
  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextFindNextGap --
 *
 *      This routine searches the closest gap starting
 *      at index for sequence i. Search to the left if
 *      index is negative, to the right otherwise.
 *
 * Results:
 *	column index of the closest gap
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextFindNextGap(register Biotext *BiotextPtr,
		   int iseq,
		   int icol) 
{
  int i, res;
  
  i = abs(icol);
  res = 0;
  while (BiotextPtr->SeqMat[iseq][i] != '.') {
    if (icol < 0) {
      i--;
      if (i < 0) {
	res = -1;
	break;
      }
    } else {    
      i++;
      if (i == BiotextPtr->lgSeqs) {
	res = BiotextPtr->lgSeqs;
	break;
      }
    }
  }

  if (! res) 
    return i;
  else 
    return res;
}

/*--------------------------------------------------------------
 *
 * BiotextPushInSeq --
 *
 *      This routine "pushes" sequences: it removes or
 *      insets a gap at imax , and removes/inserts a 
 *      gap at icol , depending if imax > icol or not.
 *
 * Results:
 *	column index of the closest gap
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
void
BiotextPushInSeq(register Biotext *BiotextPtr,
	       int iseq,
	       int icol,
	       int imax) 
{
  int i, itag;
  
  /* tag for insertion character */
  itag = 0;
  for (i=0;i<BiotextPtr->nbMapping;i++) {
    if (BiotextPtr->TagMap[i].smap == '.') {
      itag = i;
      break;
    }
  }

  /* push to the left */
  if (imax < icol) {
    for(i=imax;i<icol;i++) {
      BiotextPtr->SeqMat[iseq][i] = BiotextPtr->SeqMat[iseq][i+1];
      BiotextPtr->SeqT[iseq][i] = BiotextPtr->SeqT[iseq][i+1];
    }
  } else {
    for(i=imax;i>icol;i--) {
      BiotextPtr->SeqMat[iseq][i] = BiotextPtr->SeqMat[iseq][i-1];
      BiotextPtr->SeqT[iseq][i] = BiotextPtr->SeqT[iseq][i-1];
    }
  }
  BiotextPtr->SeqMat[iseq][icol] = '.';
  BiotextPtr->SeqT[iseq][icol] = itag;

  return;
}
    
/*--------------------------------------------------------------
 *
 * BiotextComputeGeometry --
 *
 *      This routine adjusts and requests for the 
 *      biotext widget. Size should correspond to an 
 *      integer number of characters.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
void
BiotextComputeGeometry(Biotext *BiotextPtr) 
{
  int width, height, bdi, nC, nR;

  bdi = 2*(BiotextPtr->borderWidth+BiotextPtr->inset);
  nC  = BiotextPtr->widthChar;
  nR  = BiotextPtr->heightChar;

  //fprintf(stderr, "\nBiotextComputeGeometry enter\n");
  //fprintf(stderr, "bdi %d nC %d nR %d wdt %d hgt %d\n",bdi, nC, nR, BiotextPtr->width, BiotextPtr->height);
  //fflush(stderr);
  width  = nC * BiotextPtr->charWidth  + bdi;
  height = nR * BiotextPtr->charHeight + bdi;

  //fprintf(stderr,"width %d height %d\n",width, height);

  if (Tk_IsMapped(BiotextPtr->tkwin)) {
    BiotextPtr->width  = width;
    BiotextPtr->height = height;
    Tk_GeometryRequest(BiotextPtr->tkwin, width, height);
  } else {
    BiotextPtr->width  = -1;
    BiotextPtr->height = -1;
  }
  
  //fprintf(stderr,"BiotextComputeGeometry done ...\n");

  return;
}

    
/*--------------------------------------------------------------
 *
 * BiotextSetFont --
 *
 *      This routine sets the font according to 
 *      FontFamily, FontSize and FontWeight stored in
 *      the Biotext structure.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextSetFont(register Biotext *BiotextPtr) 
{
  char *fontStr, *fnStr, *stint, *fxStr;
  Tcl_DString sfont, sxlfd, sxft;

#if DEBUG
  //fprintf(stderr, "faml %s\n",BiotextPtr->FontFamily);
  //fprintf(stderr, "size %d\n",BiotextPtr->FontSize);
  //fprintf(stderr, "wght %s\n",BiotextPtr->FontWeight);
  //fflush(stderr);
#endif

  Tcl_DStringInit(&sfont);
  Tcl_DStringInit(&sxlfd);
  stint = (char *)ckalloc(3 * sizeof(char));
  sprintf(stint, "%d", BiotextPtr->FontSize);

  /*
   * Font name as "courier 12 bold"
   */
  Tcl_DStringAppend(&sfont, BiotextPtr->FontFamily, -1);
  Tcl_DStringAppend(&sfont, " ", -1);
  Tcl_DStringAppend(&sfont, stint, -1);
  if (strcmp(BiotextPtr->FontWeight, "normal")) {
    Tcl_DStringAppend(&sfont, " ", -1);
    Tcl_DStringAppend(&sfont, BiotextPtr->FontWeight, -1);
  }

  /*
   * Font XLFD
   */
  Tcl_DStringAppend(&sxlfd, "-*-", -1);
  Tcl_DStringAppend(&sxlfd, BiotextPtr->FontFamily, -1);
  if (strcmp(BiotextPtr->FontWeight, "normal") == 0) 
    Tcl_DStringAppend(&sxlfd, "-medium-r-normal--",-1);
  else
    Tcl_DStringAppend(&sxlfd, "-bold-r-normal--",-1);
  Tcl_DStringAppend(&sxlfd, stint, -1);
  Tcl_DStringAppend(&sxlfd, "-*-*-*-*-*-iso8859-1", -1);

#ifdef USE_XFT
  /*
   * Xft font as "Courier-12:bold"
   */
  if (BiotextPtr->xft) {
    Tcl_DStringInit(&sxft);
    Tcl_DStringAppend(&sxft, BiotextPtr->FontFamily, -1);
    Tcl_DStringAppend(&sxft,"-", -1);
    Tcl_DStringAppend(&sxft, stint, -1);
    if (strcmp(BiotextPtr->FontWeight, "normal")) {
      Tcl_DStringAppend(&sxft,":", -1);
      Tcl_DStringAppend(&sxft,BiotextPtr->FontWeight, -1);
    }
    fxStr = strdup(Tcl_DStringValue(&sxft));
    BiotextPtr->FontXft = strdup(fxStr);
    Tcl_DStringFree(&sxft);

#ifndef WIN32    
    if (BiotextPtr->xftFont != NULL) 
      XftFontClose(BiotextPtr->display, BiotextPtr->xftFont);
    BiotextPtr->xftFont = XftFontOpenName(BiotextPtr->display, Tk_ScreenNumber(BiotextPtr->tkwin), BiotextPtr->FontXft);
#endif
  }
#endif

  fontStr = strdup(Tcl_DStringValue(&sxlfd));
  BiotextPtr->FontXlfd = strdup(fontStr);
  Tcl_DStringFree(&sxlfd);

  fnStr = strdup(Tcl_DStringValue(&sfont));
  BiotextPtr->FontName = strdup(fnStr);
  Tcl_DStringFree(&sfont);

  /* Now alloc the font */
  if (BiotextPtr->tkfont != NULL) 
    Tk_FreeFont(BiotextPtr->tkfont);
  BiotextPtr->tkfont = Tk_AllocFontFromObj(BiotextPtr->interp, BiotextPtr->tkwin, Tcl_NewStringObj(fnStr, -1));
  BiotextPtr->font = Tk_FontId(BiotextPtr->tkfont);

#if DEBUG
  //fprintf(stderr, "font allocation done ...\n");
#endif

  fprintf(stderr,"avant fontInfo\n");
  fflush(stderr);
  BiotextFontInfo(BiotextPtr);

#if DEBUG
  //fprintf(stderr, "BiotextSetFont done ...\n");
#endif

  return TCL_OK;
}
    
/*--------------------------------------------------------------
 *
 * BiotextInitBuffer --
 *
 *      This routine initialises or resets the buffer
 *      used to copy/cut/paste the sequences.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextInitBuffer(register Biotext *BiotextPtr,
		  int nseq) 
{
  int i;

  if (BiotextPtr->nbrSeqsCache != 0) {
    ckfree((char *)BiotextPtr->SeqGrpCache);
    for (i=0; i<BiotextPtr->nbrSeqsCache; i++) {
      ckfree((char *)BiotextPtr->SeqMatCache[i]);
      ckfree((char *)BiotextPtr->SeqTCache[i]);
    }
    ckfree((char *)BiotextPtr->SeqMatCache);
    ckfree((char *)BiotextPtr->SeqTCache);
  }
  BiotextPtr->nbrSeqsCache = 0;

  BiotextPtr->SeqGrpCache = (int *)ckalloc((nseq+1)*sizeof(int));
  BiotextPtr->SeqMatCache = (char **)ckalloc((nseq+1)*sizeof(char *));
  BiotextPtr->SeqTCache   = (int **)ckalloc((nseq+1)*sizeof(int *));
  for (i=0;i<nseq;i++) {
    BiotextPtr->SeqMatCache[i] = ckalloc(BiotextPtr->lgAlloc*sizeof(char));
    BiotextPtr->SeqTCache[i]   = (int *)ckalloc(BiotextPtr->lgAlloc*sizeof(int));
  }
  BiotextPtr->lgSeqsCache = BiotextPtr->lgSeqs;
  BiotextPtr->SeqMatCache[nseq] = NULL;
  BiotextPtr->SeqTCache[nseq]   = NULL;

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextCopySeqsToBuffer --
 *
 *      This routine copies a sequence given its index
 *      to the copy/paste buffer. It copies all the
 *      attributes of the sequences.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextCopySeqsToBuffer(register Biotext *BiotextPtr,
			int nseqs,
			Tcl_Obj **OIdx) 
{  
  int i, j, idx;

  for (i=0;i<nseqs;i++) {
    Tcl_GetIntFromObj(BiotextPtr->interp,OIdx[i], &idx);
    for (j=0;j<BiotextPtr->lgSeqs;j++) {
      BiotextPtr->SeqMatCache[i][j] = BiotextPtr->SeqMat[idx][j];
      BiotextPtr->SeqTCache[i][j] = BiotextPtr->SeqT[idx][j];
    }
    BiotextPtr->SeqGrpCache[i] = BiotextPtr->SeqGrp[idx];
  }

  BiotextPtr->nbrSeqsCache = nseqs;

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextAddColsToBufferSeqs --
 *
 *      This routines adds cols to the sequences 
 *      contained in the paste/copy buffer.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextAddColsToBufferSeqs(register Biotext *BiotextPtr,
			   int ncols) 
{
  int i, j, itag;

  /* realloc to the actual sequence length */
  for (i=0;i<BiotextPtr->nbrSeqsCache;i++) {
    BiotextPtr->SeqMatCache[i] = (char *)ckrealloc((char *)BiotextPtr->SeqMatCache[i],BiotextPtr->lgAlloc*sizeof(char));
    BiotextPtr->SeqTCache[i]   = (int *)ckrealloc((char *)BiotextPtr->SeqTCache[i],BiotextPtr->lgAlloc*sizeof(int));
  }
  
  /* search for gap tag index */
  itag = 0;
  for (i=0;i<BiotextPtr->nbMapping;i++) {
    if (BiotextPtr->TagMap[i].smap == '.') {
      itag = i;
      break;
    }
  }

  /* fill added columns with gaps */
  for(i=0;i<BiotextPtr->nbrSeqsCache;i++) {
    for(j=BiotextPtr->lgSeqsCache;j<BiotextPtr->lgSeqsCache+ncols;j++) {
      BiotextPtr->SeqMatCache[i][j] = '.';
      BiotextPtr->SeqTCache[i][j] = itag;
    }
  }

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * BiotextFontInfo --
 *      Gives font information. Only use non-Xft fonts
 *
 * Results:
 *      None. 
 *
 * Side effects:
 *      None.      
 *
 *--------------------------------------------------------------
 */
void
BiotextFontInfo(Biotext *BiotextPtr) 
{
  Tk_FontMetrics fm;
  Tk_Font tkfont;

  tkfont = BiotextPtr->tkfont;
  Tk_GetFontMetrics(tkfont, &fm);
  BiotextPtr->charWidth = Tk_TextWidth(tkfont, "Z", 1);
  BiotextPtr->charHeight = fm.ascent+fm.descent;
  BiotextPtr->ascent     = fm.ascent;
  BiotextPtr->descent    = fm.descent;

  return;
}

void DumpSeqs(Biotext *BiotextPtr) 
{
  int i,j,lg;

  lg = BiotextPtr->lgSeqs;
  for (i=0; i<BiotextPtr->nbSeqs;i++) {
    fprintf(stderr,"i= %d %d s=>>",i, (int)BiotextPtr->SeqMat[i][0]);
    for (j=0;j<10;j++) 
      fprintf(stderr, "%c",BiotextPtr->SeqMat[i][j]);
    fprintf(stderr,"<< >>");
    for (j=lg-10;j<lg;j++) 
      fprintf(stderr, "%c",BiotextPtr->SeqMat[i][j]);
    fprintf(stderr,"<<\n");
  }
  fflush(stderr);

  return;
}




