#include "tkBiotext.h"

/*
 *--------------------------------------------------------------
 *
 * BiotextSelectionCmd --
 *
 *	This procedure is invoked to process the 
 *      "selection" method. It
 *	dispatches the sub-methods of "selction".
 *      Valid options are : from, to, clear
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextSelectionCmd(Biotext *BiotextPtr,
		    Tcl_Interp *interp,
		    int objc,
		    Tcl_Obj * const objv[]) 
{

  int cmdIndex, r, c;

  static CONST84 char *selCmdNames[] = {
    "clear", "from", "to", (char *)NULL
  };
  enum selCommand {
    CMD_SEL_CLEAR, CMD_SEL_FROM, CMD_SEL_TO,
  };

  if (Tcl_GetIndexFromObj(interp, objv[2], selCmdNames, "selection option", 0, &cmdIndex) != TCL_OK) {
    return TCL_ERROR;
  }
  switch ((enum selCommand) cmdIndex) {
  case CMD_SEL_CLEAR:
    BiotextPtr->selFirstX = -1;
    BiotextPtr->selFirstY = -1;
    BiotextPtr->selLastX  = -1;
    BiotextPtr->selLastY  = -1;
    break;
  case CMD_SEL_FROM:
    if (objc != 4) {
      return TCL_ERROR;
    }
    BiotextGetIndexObj(BiotextPtr,objv[3], &r, &c);

    BiotextPtr->selFirstX = r;
    BiotextPtr->selFirstY = c;
    break;
  case CMD_SEL_TO:
    if (objc != 4) {
      return TCL_ERROR;
    }
    BiotextGetIndexObj(BiotextPtr,objv[3], &r, &c);

    BiotextPtr->selLastX  = r;
    BiotextPtr->selLastY  = c;
  }
  
  BiotextEventuallyRedraw(BiotextPtr);
  
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * BiotextAddSeqsCmd --
 *
 *	This procedure is invoked to process the 
 *      "addseqs" method. It
 *	adds a list of sequences to the widget,
 *      and display them. If possible, it maps
 *      each residue.
 *      Be aware that NO checking on the sequences is
 *      done at this level !
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextAddSeqsCmd(register Biotext *BiotextPtr,
		  Tcl_Interp *interp,
		  int objc,
		  Tcl_Obj * const objv[]) 
{
  int i, j, k, nseq, slen, index, diff;
  char *iStr, dummy, *seq;
  Tcl_Obj **OSeqMat;
  
  /* Sequences should already be present */
  if (BiotextPtr->nbSeqs <= 0) return TCL_OK;

  /* parse arguments */
  if (objc < 4) {
    Tcl_WrongNumArgs(interp, 1, objv, "addseqs seq-list index");
    return TCL_ERROR;
  }

  iStr = Tcl_GetString(objv[3]);
  if (strcmp(iStr, "end") && (sscanf(iStr, "%d%s", &index, &dummy) != 1)) {
    Tcl_WrongNumArgs(interp, 1, objv, "addseqs <seq-list> index");
    return TCL_ERROR;
  }
  if (strcmp(iStr, "end") == 0) 
    index = BiotextPtr->nbSeqs;

  Tcl_ListObjGetElements(interp,objv[2],&nseq,&OSeqMat);
  if (nseq == 0) return TCL_OK;
  /* Get inserted sequence length */
  Tcl_GetByteArrayFromObj(OSeqMat[0],&slen);

  diff = BiotextPtr->lgSeqs - slen;
  if (diff < 0) {
    /* Inserted sequences are longer than the one
     * in the alignment. Insert cols at the end.
     */
    diff = abs(diff);
    if (BiotextInsertCols(BiotextPtr, BiotextPtr->lgSeqs-1, diff) != TCL_OK) return TCL_ERROR;
  }
  
  /*
   * Insert empty rows after index
   * record previous number of seqs 
   */
  BiotextInsertRows(BiotextPtr, index, nseq);
  /* fill alignement with new sequence */
  for(k=0,i=index+1;i<index+1+nseq;i++,k++) {
    seq = Tcl_GetString(OSeqMat[k]);
    for(j=0;j<slen;j++)
      BiotextPtr->SeqMat[i][j] = seq[j];
  }
  
  /* initialise tag for new seq */
  for (i=index+1;i<index+1+nseq;i++) 
    for (j=0;j<slen;j++) 
      BiotextPtr->TagSeq[i][j] = -1;

  /* map the new sequences */
  if (BiotextPtr->Mapping != NULL) {
    for (i=index+1;i<index+1+nseq;i++) {
      for (j=0;j<slen;j++) {
	for (k=0;k<BiotextPtr->nbMapping;k++) {
	  if (BiotextPtr->SeqMat[i][j] == BiotextPtr->Mapping[k].smap) {
	    BiotextPtr->MapSeq[i][j] = k;
	    break;
	  }
	}
      }
    }
  }

  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * BiotextSequencesCmd --
 *
 *	This procedure is invoked to process the 
 *      "sequences" method.
 *	It uploads a list of sequences into the widget,
 *      and displays them. If possible, it assign a tag
 *      to each residue.
 *      Be aware that no checking on the sequences is
 *      done at this level !
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
a */
int 
BiotextSequencesCmd(Biotext *BiotextPtr,
		    Tcl_Interp *interp,
		    int objc,
		    Tcl_Obj * const objv[]) 
{
  int i, j, k, nseq, slen;
  Tcl_Obj **OSeqMat;

  Tcl_ListObjGetElements(interp,objv[2],&nseq,&OSeqMat);
  if (nseq == 0) return TCL_OK;

  /*
   * Arrays allocation 
   */
  BiotextPtr->SeqMat = (char **)ckalloc(nseq*sizeof(char *));
  /* sequence length , WITHOUT \0 */
  Tcl_GetByteArrayFromObj(OSeqMat[0],&slen);

  /* Allocate and fill sequences array */
  for (i=0;i<nseq;i++) {
    BiotextPtr->SeqMat[i] = (char *)ckalloc((slen+1)*sizeof(char));
    strcpy(BiotextPtr->SeqMat[i],Tcl_GetString(OSeqMat[i]));
  }
  BiotextPtr->nbSeqs  = nseq;
  BiotextPtr->lgSeqs  = slen;
  BiotextPtr->lgAlloc = slen;

  /* sequence groupe number */
  BiotextPtr->SeqGrp = (int *)ckalloc(nseq*sizeof(int));
  for (i=0;i<nseq;i++) {
    BiotextPtr->SeqGrp[i] = 0;
  }

  /* initialise tags if necessary */
  BiotextPtr->Tags = (Tag *)ckalloc(1*sizeof(Tag));

  /* initialize array for sequence tagging */
  BiotextPtr->TagSeq = (int **)ckalloc(nseq*sizeof(int *));
  for (i=0;i<nseq;i++) {
    BiotextPtr->TagSeq[i] = (int *)ckalloc(slen*sizeof(int));
  }
  RemoveTaggedPositionsInAlignment(BiotextPtr,-1);

  /* mapping alignment elements */
  if (BiotextPtr->Mapping != NULL) {
    BiotextPtr->MapSeq = (int **)ckalloc(nseq*sizeof(int *));
    for (i=0;i<nseq;i++) {
      BiotextPtr->MapSeq[i] = (int *)ckalloc(slen*sizeof(int));
    }
    /* fill mapping array */
    for (i=0;i<nseq;i++) {
      for (j=0;j<slen;j++) {
	for (k=0;k<BiotextPtr->nbMapping;k++) {
	  if (BiotextPtr->SeqMat[i][j] == BiotextPtr->Mapping[k].smap) {
	    BiotextPtr->MapSeq[i][j] = k;
	    break;
	  }
	}
      }
    }
  }

  /* Redraw the widget */
  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);
  
  return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * BiotextMappingCmd --
 *
 *	This procedure is invoked to process the 
 *      "mapping" method. It
 *	defines a new mapping as a list of 
 *      {key foreground background} items
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextMappingCmd(Biotext *BiotextPtr,
		  Tcl_Interp *interp,
		  int objc,
		  Tcl_Obj * const objv[]) 
{
  char *tmp;
  int i, j, k, nmap;
  Tcl_Obj **OMap, **OneE ;

  if (objc != 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg...?");
    return TCL_ERROR;
  }

  Tcl_ListObjGetElements(interp,objv[2],&nmap,&OMap);
  BiotextPtr->nbMapping=nmap;
  if (BiotextPtr->Mapping != NULL) 
    ckfree((char *)BiotextPtr->Mapping);
  BiotextPtr->Mapping = (Tag *)ckalloc(nmap*sizeof(Tag));
  
  for (i=0;i<nmap;i++) {
    Tcl_ListObjGetElements(interp,OMap[i],&j,&OneE);
    tmp = Tcl_GetString(OneE[0]);
    BiotextPtr->Mapping[i].smap=tmp[0];
    BiotextPtr->Mapping[i].fg  = Tk_3DBorderColor(Tk_Alloc3DBorderFromObj(interp,BiotextPtr->tkwin,OneE[1]))->pixel;
    BiotextPtr->Mapping[i].bg  = Tk_3DBorderColor(Tk_Alloc3DBorderFromObj(interp,BiotextPtr->tkwin,OneE[2]))->pixel;
    BiotextPtr->Mapping[i].bgBorder = Tk_Get3DBorderFromObj(BiotextPtr->tkwin, OneE[2]);
  }

  Biotext_AssignColors(BiotextPtr, 2);

  if (BiotextPtr->SeqMat != NULL) {
    if (BiotextPtr->MapSeq == NULL) {
      BiotextPtr->MapSeq = (int **)ckalloc(BiotextPtr->nbSeqs * sizeof(int *));
      for (i=0;i<BiotextPtr->nbSeqs;i++) {
	BiotextPtr->MapSeq[i] = (int *)ckalloc(BiotextPtr->lgSeqs * sizeof(int));
      }
    }

    int pt;
    for (i=0;i<BiotextPtr->nbSeqs;i++) {
      for (j=0;j<BiotextPtr->lgSeqs;j++) {
	pt = 0;
	for (k=0;k<BiotextPtr->nbMapping;k++) {
	  if (BiotextPtr->SeqMat[i][j] == BiotextPtr->Mapping[k].smap) {
	    BiotextPtr->MapSeq[i][j] = k;
	    pt = 1;
	    break;
	  }
	}
	if (! pt) {
	  for (j=0;j<BiotextPtr->lgSeqs;j++) {
	    if (BiotextPtr->MapSeq[i][j] < 0 || BiotextPtr->MapSeq[i][j] >= BiotextPtr->nbMapping) {
	      break;
	    }
	  }
	}
      }
    }
  }
  
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * BiotextMapCmd --
 *
 *	This procedure is invoked to process the 
 *      "map" method. It enables/disables the
 *	drawing of mapped characters. 
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextMapCmd(Biotext *BiotextPtr,
	      Tcl_Interp *interp,
	      int objc,
	      Tcl_Obj * const objv[]) 
{
  int bool, i, j;

  if (objc != 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg...?");
    return TCL_ERROR;
  }
  
  /* argument could be map on|off|clean */
  if (strcmp(Tcl_GetString(objv[2]), "clean") == 0) {
    /* clean mapping */
    for (i=0;i<BiotextPtr->nbSeqs;i++) {
      for (j=0;j<BiotextPtr->lgSeqs;j++) {
	BiotextPtr->MapSeq[i][j] = -1;
      }
    }
  } else {
    Tcl_GetBooleanFromObj(interp,objv[2],&bool);
    if (bool) {
      if (BiotextPtr->flags & MAP_RESIDUES) {
	return TCL_OK;
      } else {
	BiotextPtr->flags |= MAP_RESIDUES;
      }
    } else {
      if (BiotextPtr->flags & MAP_RESIDUES) {
	BiotextPtr->flags &= ~MAP_RESIDUES;
      } else {
	return TCL_OK;
      }
    }    
  }

  BiotextEventuallyRedraw(BiotextPtr);
  
  return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * BiotextTagCmd --
 *
 *	This procedure is invoked to process the 
 *      "tag" method. It dispatches the sub-commands 
 *      configure, add and remove
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextTagCmd(Biotext *BiotextPtr,
	      Tcl_Interp *interp,
	      int objc,
	      Tcl_Obj * const objv[]) 
{
  int cmdIndex, i, j, k, iTag, row1, col1, row2, col2, tmp, bool;
  char *tagName, *ind1, *ind2;
  Tk_3DBorder bgBorder;
  unsigned long fg, bg;

  static CONST84 char *tagCmdNames[] = {
    "configure", "add", "remove", "delete", "state", (char *)NULL
  };
  enum tagCommand {
    CMD_TAG_CONFIG, CMD_TAG_ADD, CMD_TAG_REMOVE, CMD_TAG_DELETE, CMD_TAG_STATE,
  };

  if (Tcl_GetIndexFromObj(interp, objv[2], tagCmdNames, "tag option", 0, &cmdIndex) != TCL_OK) {
    return TCL_ERROR;
  }

  switch ((enum tagCommand) cmdIndex) {
  case CMD_TAG_CONFIG:
    if (objc != 8) {
      Tcl_WrongNumArgs(interp, 1, objv, "tag configure tagName -foreground fgCol -background bgCol");
      return TCL_ERROR;
    }

    tagName = Tcl_GetString(objv[3]);
    /* Now take the tag definition */
    bg = fg = -1;
    bgBorder = NULL;
    for (i=4;i<7;i+=2) {
      if (strcasecmp(Tcl_GetString(objv[i]),"-foreground") == 0 || strcasecmp(Tcl_GetString(objv[i]),"-fg") == 0) {
	fg  = Tk_3DBorderColor(Tk_Alloc3DBorderFromObj(interp,BiotextPtr->tkwin,objv[i+1]))->pixel;
      }
      if (strcasecmp(Tcl_GetString(objv[i]),"-background") == 0 || strcasecmp(Tcl_GetString(objv[i]),"-bg") == 0) {
	bg  = Tk_3DBorderColor(Tk_Alloc3DBorderFromObj(interp,BiotextPtr->tkwin,objv[i+1]))->pixel;
	bgBorder = Tk_Get3DBorderFromObj(BiotextPtr->tkwin,objv[i+1]);
      }
    }
    
    /* check if tag already exists */
    iTag=0;
    for (i=0;i<BiotextPtr->nbTags;i++) {
      if (strcmp(tagName,BiotextPtr->Tags[i].tagName) == 0) 
	break;
    }
    iTag = i;
    if (iTag == BiotextPtr->nbTags) {
      /* allocate space for new tag */
      BiotextPtr->nbTags++;
      BiotextPtr->Tags = (Tag *)ckrealloc((char *)BiotextPtr->Tags,(BiotextPtr->nbTags+1)*sizeof(Tag));
    }

    /* check and allocate if necessary the tag sequences array */
    if (BiotextPtr->TagSeq == NULL) {
      BiotextPtr->TagSeq = (int **)ckalloc(BiotextPtr->nbSeqs*sizeof(int *));
      for (i=0;i<BiotextPtr->nbSeqs;i++) 
	BiotextPtr->TagSeq[i] = (int *)ckalloc(BiotextPtr->lgSeqs*sizeof(int));
      /* now initialize */
      RemoveTaggedPositionsInAlignment(BiotextPtr,-1);
    }
    if (bg == -1 || fg == -1) 
      return TCL_ERROR;

    /* fill the tag structure */
    BiotextPtr->Tags[iTag].tagName=(char *)ckalloc((strlen(tagName)+1)*sizeof(char));
    strcpy(BiotextPtr->Tags[iTag].tagName,tagName);

    BiotextPtr->Tags[iTag].fg  = fg;
    BiotextPtr->Tags[iTag].bg  = bg;
    BiotextPtr->Tags[iTag].bgBorder = bgBorder;

    Biotext_AssignColors(BiotextPtr, 3);
      
    break;
  case CMD_TAG_ADD:
    tagName = Tcl_GetString(objv[3]);
    /* check if tag exists */
    for(i=0;i<BiotextPtr->nbTags;i++) {
      if (strcmp(BiotextPtr->Tags[i].tagName,tagName) == 0)
	break;
    }
    if (i==BiotextPtr->nbTags)
      return TCL_ERROR;
    iTag=i;

    /* now check the indexes -
     * There must be an even number of indexes
     */
    if (((objc-4) % 2) != 0)
      return TCL_ERROR;
    /* read the indexes, and store in the tag matrix */
    for (j=4;j<objc;j+=2) {
      ind1 = Tcl_GetString(objv[j]);
      BiotextGetIndex(BiotextPtr, ind1, &row1, &col1);
      ind2 = Tcl_GetString(objv[j+1]);
      BiotextGetIndex(BiotextPtr, ind2, &row2, &col2);

      tmp = row1;
      row1 = MIN(row1,row2);
      row2 = MAX(tmp,row2);
      tmp = col1;
      col1 = MIN(col1,col2);
      col2 = MAX(tmp,col2);

      /* define rectangle defined by :
       * row1/col1 (upper left corner) and 
       * row2/col2 (bottom right corner)
       */
      for(i=row1;i<=row2;i++) 
	for(k=col1;k<=col2;k++)
	  BiotextPtr->TagSeq[i][k]=iTag;
    }

    break;
  case CMD_TAG_REMOVE:
    if (objc == 3) {
      /* remove all tags */
      RemoveTaggedPositionsInAlignment(BiotextPtr,-1);
    } else {
      /* remove a specific tag */
      tagName = Tcl_GetString(objv[4]);
      for (i=0;i<BiotextPtr->nbTags;i++) 
	if (strcmp(BiotextPtr->Tags[i].tagName,tagName) == 0)
	  break;
      if (i==BiotextPtr->nbTags)
	return TCL_ERROR;

      iTag=i;
      RemoveTaggedPositionsInAlignment(BiotextPtr,iTag);
    }
    break;
  case CMD_TAG_DELETE:
    if (objc == 3) {
      /* delete all tags */
      ckfree((char *)BiotextPtr->Tags);
      /* remove the tagged position in the alignment */
      RemoveTaggedPositionsInAlignment(BiotextPtr,-1);
      BiotextPtr->nbTags=0;
    } else {
      /* delete a given tag */
      tagName = Tcl_GetString(objv[3]);
      for (i=0;i<BiotextPtr->nbTags;i++) 
	if (strcmp(BiotextPtr->Tags[i].tagName,tagName) == 0)
	  break;
      if (i==BiotextPtr->nbTags)
	return TCL_ERROR;

      iTag=i;
      BiotextPtr->Tags[iTag].tagName[0]='\0';
      /* delete tagged positions */
      RemoveTaggedPositionsInAlignment(BiotextPtr,iTag);
    }      
    break;
  case CMD_TAG_STATE:
    if (objc != 4) {
      Tcl_WrongNumArgs(interp, 1, objv, "tag state boolean");
      return TCL_ERROR;
    }
    
    Tcl_GetBooleanFromObj(interp,objv[3],&bool);
    if (bool) {
      if (! (BiotextPtr->flags & TAG_ALIGNMENT)) {
	BiotextPtr->flags |= TAG_ALIGNMENT;
      }
    } else {
      if (BiotextPtr->flags & TAG_ALIGNMENT) {
	BiotextPtr->flags &= ~TAG_ALIGNMENT;
      }
    }
    break;
  }
  
  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);

  return TCL_OK;
}


/*--------------------------------------------------------------
 *
 * BiotextXViewCmd --
 *
 *	This procedure is invoked to process the 
 *      "xview" method. It update the widget in
 *	accordance to x-scrolling.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int
BiotextXViewCmd(register Biotext *BiotextPtr,
		Tcl_Interp *interp,
		int objc,
		Tcl_Obj * const objv[]) 
{
  int index, charsPerPage;

  index = charsPerPage = 0;
  if (objc == 2) {
    double first, last;
    char buf[TCL_DOUBLE_SPACE];
    
    BiotextVisibleXRange(BiotextPtr, &first, &last);
    Tcl_PrintDouble(NULL, first, buf);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    Tcl_PrintDouble(NULL, last, buf);
    Tcl_AppendResult(interp, " ", buf, NULL);

    return TCL_OK;
  } else if (objc == 3) {
    /* TO DO 
       if (BiotextGetIndex(BiotextPtr, Tcl_GetString(objv[2]), &index) != TCL_OK) {
       goto error;
       }
    */
  } else {
    double fraction;
    int count;
    
    charsPerPage = BiotextPtr->widthChar;
    index = BiotextPtr->leftIndex;

    switch (Tk_GetScrollInfoObj(interp, objc, objv, &fraction,&count)) {
    case TK_SCROLL_ERROR:
      return TCL_ERROR;
    case TK_SCROLL_MOVETO:
      index = (int) (fraction * BiotextPtr->lgSeqs);
      break;
    case TK_SCROLL_PAGES: {
      index += count * charsPerPage;
      break;
    }
    case TK_SCROLL_UNITS:
      index += count;
      break;
    }
  }
  
  if (index >= BiotextPtr->lgSeqs) {
    index = BiotextPtr->lgSeqs - 1;
  }
  if (index > BiotextPtr->lgSeqs - charsPerPage) {
    index = BiotextPtr->lgSeqs - charsPerPage;
  }
  if (index < 0) {
    index = 0;
  }
  BiotextPtr->leftIndex = index;

  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);
  
  return TCL_OK;
}


/*--------------------------------------------------------------
 *
 * BiotextYViewCmd --
 *
 *	This procedure is invoked to process the 
 *      "yview" method. It update the widget in
 *	accordance to y-scrolling.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextYViewCmd(register Biotext *BiotextPtr,
		Tcl_Interp *interp,
		int objc,
		Tcl_Obj * const objv[]) 
{
  int index, charsPerPage;
  
  index = charsPerPage = 0;
  if (objc == 2) {
    double first, last;
    char buf[TCL_DOUBLE_SPACE];
    
    BiotextVisibleYRange(BiotextPtr, &first, &last);
    Tcl_PrintDouble(NULL, first, buf);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    Tcl_PrintDouble(NULL, last, buf);
    Tcl_AppendResult(interp, " ", buf, NULL);

    return TCL_OK;
  } else if (objc == 3) {
    /* TO DO */
  } else {
    double fraction;
    int count;
    
    charsPerPage = BiotextPtr->heightChar;
    index = BiotextPtr->topIndex;

    switch (Tk_GetScrollInfoObj(interp, objc, objv, &fraction,&count)) {
    case TK_SCROLL_ERROR:
      return TCL_ERROR;
    case TK_SCROLL_MOVETO:
      index = (int) (fraction*BiotextPtr->nbSeqs + 0.5);
      break;
    case TK_SCROLL_PAGES: {
      index += count * charsPerPage;
      break;
    }
    case TK_SCROLL_UNITS:
      index += count;
      break;
    default:
      return TCL_ERROR;
    }
  }
  if (index >= BiotextPtr->nbSeqs) {
    index = BiotextPtr->nbSeqs - 1;
  }
  if (index >= BiotextPtr->nbSeqs - charsPerPage ) {
    index = BiotextPtr->nbSeqs - charsPerPage;
  }
  if (index < 0) {
    index = 0;
  }
  BiotextPtr->topIndex = index;

  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);
  
  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextInsertCmd --
 *
 *	This procedure is invoked to process the 
 *      "insert" method. It inserts columns of 
 *      gaps or sequences after or before index , 
 *      depending on the sign of count.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextInsertCmd(register Biotext *BiotextPtr,
		Tcl_Interp *interp,
		int objc,
		Tcl_Obj * const objv[]) 
{
  char *iStr, dummy;
  int i, result, count, cmdIndex;

  if (objc < 4) {
    Tcl_WrongNumArgs(interp, 1, objv, "option index ?count?");
    return TCL_ERROR;
  }

  static CONST84 char *insCmdNames[] = {
    "cols", "rows", (char *)NULL
  };
  enum insCommand {
    ICMD_COLS, ICMD_ROWS,
  };

  if (Tcl_GetIndexFromObj(interp, objv[2], insCmdNames, "insert option", 0, &cmdIndex) != TCL_OK) {
    return TCL_ERROR;
  }

  count = 1; /* default */
  if (objc == 5) {
    if (Tcl_GetInt(interp, Tcl_GetString(objv[4]), &count) != TCL_OK) {
      return TCL_ERROR;
    }
  }

  /* Get insertion index */
  iStr = Tcl_GetString(objv[3]);
  if (strcmp(iStr, "end") && (sscanf(iStr, "%d%s", &i, &dummy) != 1)) {
    Tcl_WrongNumArgs(interp, 1, objv, "option index ?count?");
    return TCL_ERROR;
  }
  
  switch ((enum insCommand) cmdIndex) {
  case ICMD_COLS:
    if (strcmp(iStr, "end") == 0) {
      i = BiotextPtr->lgSeqs - 1;
    } else {
      sscanf(iStr, "%d", &i);
    }
    result = BiotextInsertCols(BiotextPtr, i, count);
    break;
  case ICMD_ROWS:
    if (strcmp(iStr, "end") == 0) {
      i = BiotextPtr->nbSeqs - 1;
    } else {
      sscanf(iStr, "%d", &i);
    }
    result = BiotextInsertRows(BiotextPtr, i, count);
    break;
  default:
    Tcl_WrongNumArgs(interp, 1, objv, "option index ?count?");
    result = TCL_ERROR;
  }

  if (result == TCL_OK) {
    BiotextPtr->flags |= UPDATE_SCROLLBAR;
    BiotextEventuallyRedraw(BiotextPtr);
  }

  return result;
}

/*--------------------------------------------------------------
 *
 * BiotextCharsCmd --
 *
 *      This procedure implements the "chars" method.
 *      It inserts <count> character <char> in 
 *      biotext at index <i>.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextCharsCmd(register Biotext *BiotextPtr,
		  Tcl_Interp *interp,
		  int objc,
		  Tcl_Obj * const objv[]) 
{
  char *chr, *index;
  int count, r, c, nGrp, i, j, ichr, nogroup, iChar;

  if (objc < 4) {
    Tcl_WrongNumArgs(interp, 1, objv, "?-nogroup? <char> <index> ?count?");
    return TCL_ERROR;
  }

  /* 
   * Command : 
   * $w chars ?-nogroup? <char> <index> ?<count>?
   */
  nogroup = 0;
  iChar = 2;
  if (strcasecmp(Tcl_GetString(objv[2]),"-nogroup") == 0) {
    /* -nogroup option */
    nogroup = 1;
    iChar = 3;

    if (objc < 5) {
      Tcl_WrongNumArgs(interp, 1, objv, "?-nogroup? <char> <index> ?count?");
      return TCL_ERROR;
    }
  }

  chr = Tcl_GetString(objv[iChar]);
  if (strlen(chr) > 1) 
    return TCL_ERROR;
  if (chr[0] != '.' && BiotextPtr->Lock) 
    return TCL_OK;
  chr[0] = toupper(chr[0]);

  /* Get insertion index */
  index = Tcl_GetString(objv[iChar+1]);
  if (BiotextGetIndex(BiotextPtr, index, &r, &c) != TCL_OK) {
    return TCL_ERROR;
  }

  /* Get number of chars to insert */
  count = 1;
  if (objc == iChar+3) {
    if (Tcl_GetInt(interp, Tcl_GetString(objv[iChar+2]), &count) != TCL_OK) {
      return TCL_ERROR;
    }
  }
  
  /*
   * First :
   * delete the timer of the blinking cursor
   */
  if (BiotextPtr->cursorTimer != NULL) {
    Tcl_DeleteTimerHandler(BiotextPtr->cursorTimer);
  }

  /* 
   * Second :
   * Insert <count> cols at the end of the alignment
   */
  if (BiotextInsertCols(BiotextPtr, BiotextPtr->lgSeqs-1, count) != TCL_OK) {
    BiotextPtr->flags |= CURSOR_ON;
    BiotextConfigCursor(BiotextPtr);
    BiotextEventuallyRedraw(BiotextPtr);

    return TCL_ERROR;
  }

  /*
   * Third :
   * find tag for inserted character 
   */
  ichr = 0;
  for (i=0;i<BiotextPtr->nbMapping;i++) {
    if (BiotextPtr->Mapping[i].smap == chr[0]) {
      ichr = i;
      break;
    }
  }
  /* then insert chars */
  nGrp = BiotextPtr->SeqGrp[r];
  if (! nogroup && nGrp != 0 && BiotextPtr->nbrGrps > 0) {
    for(i=0;i<BiotextPtr->nbSeqs;i++) 
      if (BiotextPtr->SeqGrp[i] == nGrp) {
	for(j=BiotextPtr->lgSeqs-1;j>=c+count;j--) {
	  BiotextPtr->SeqMat[i][j] = BiotextPtr->SeqMat[i][j-count];
	  BiotextPtr->MapSeq[i][j] = BiotextPtr->MapSeq[i][j-count];
	}
	for(j=c;j<c+count;j++) {
	  BiotextPtr->SeqMat[i][j] = chr[0];
	  BiotextPtr->MapSeq[i][j] = ichr;
	}
      }
  } else {
    for(j=BiotextPtr->lgSeqs-1;j>=c+count;j--) {
      BiotextPtr->SeqMat[r][j] = BiotextPtr->SeqMat[r][j-count];
      BiotextPtr->MapSeq[r][j] = BiotextPtr->MapSeq[r][j-count];
    }
    for (j=c; j<c+count; j++) {
      BiotextPtr->SeqMat[r][j] = chr[0];
      BiotextPtr->MapSeq[r][j] = ichr;
    }
  }

  /* move cursor +count */
  BiotextPtr->currentC += count;
  
  BiotextPtr->flags |= CURSOR_ON;
  BiotextConfigCursor(BiotextPtr);
  BiotextEventuallyRedraw(BiotextPtr);

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextDelcharsCmd --
 *
 *      This procedure implements the "delchars" 
 *      method.
 *      It deletes <count> characters in 
 *      biotext before index <i>
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextDelcharsCmd(register Biotext *BiotextPtr,
		   Tcl_Interp *interp,
		   int objc,
		   Tcl_Obj * const objv[]) 
{
  char *index;
  int count, r, c, nGrp, i, j, k, isgap, igap, iidx, dogrp, hasmoved;

  if (objc < 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "delchars index");
    return TCL_ERROR;
  }

  /* 
   * Command : $w delchars ?-group? <index> ?<count>?
   */
  if (strcmp(Tcl_GetString(objv[2]),"-group") == 0) {
    iidx  = 3;
    dogrp = 1;
  } else {
    iidx  = 2;
    dogrp = 0;
  }

  index = Tcl_GetString(objv[iidx]);
  if (BiotextGetIndex(BiotextPtr, index, &r, &c) != TCL_OK)
    return TCL_ERROR;
  /* leftmost edge */
  if (c == 0) return TCL_OK;
  
  /* count : number of chars to be deleted */
  count = 1;
  if ((! dogrp && objc == 4) || (dogrp && objc == 5)) {
    if (Tcl_GetInt(interp, Tcl_GetString(objv[objc-1]), &count) != TCL_OK) {
	return TCL_ERROR;
    }
  }

  /*
   * find tag for inserted character 
   */
  igap = -1;
  for (i=0;i<BiotextPtr->nbMapping;i++) {
    if (BiotextPtr->Mapping[i].smap == '.') {
      igap = i;
      break;
    }
  }

  /*
   * First, delete the timer of the blinking cursor
   */
  if (BiotextPtr->cursorTimer != NULL) {
    Tcl_DeleteTimerHandler(BiotextPtr->cursorTimer);
  }

  /*
   * Loop <count> times and :
   * - check if there is a gap character at c-1
   * - if no, and no locking, delete char
   */
  nGrp = BiotextPtr->SeqGrp[r];
  for (k=0; k<count; k++) {
    hasmoved = 0;
    if (dogrp && nGrp != 0 && BiotextPtr->nbrGrps > 0) {
      isgap = 1;
      for(i=0;i<BiotextPtr->nbSeqs;i++) {
	if (BiotextPtr->SeqGrp[i] == nGrp) {
	  /* first, check for gap */
	  if (BiotextPtr->SeqMat[i][c-1] != '.') {
	    isgap = 0;
	    break;
	  }
	}
      }
      if (isgap || ! BiotextPtr->Lock) {
	hasmoved = 1;
	for(i=0;i<BiotextPtr->nbSeqs;i++) {
	  if (BiotextPtr->SeqGrp[i] == nGrp) {
	    for(j=c-1;j<BiotextPtr->lgSeqs-1;j++) {
	      BiotextPtr->SeqMat[i][j] = BiotextPtr->SeqMat[i][j+1];
	      BiotextPtr->MapSeq[i][j] = BiotextPtr->MapSeq[i][j+1];
	    }
	    BiotextPtr->SeqMat[i][BiotextPtr->lgSeqs-1] = '.';
	    BiotextPtr->MapSeq[i][BiotextPtr->lgSeqs-1] = igap;
	  }
	}
      }
    } else {
      if (BiotextPtr->SeqMat[r][c-1] == '.' || ! BiotextPtr->Lock) {
	hasmoved = 1;
	for(j=c-1;j<BiotextPtr->lgSeqs-1;j++) {
	  BiotextPtr->SeqMat[r][j] = BiotextPtr->SeqMat[r][j+1];
	  BiotextPtr->MapSeq[r][j] = BiotextPtr->MapSeq[r][j+1];
	}
	BiotextPtr->SeqMat[r][BiotextPtr->lgSeqs-1] = '.';
	BiotextPtr->MapSeq[r][BiotextPtr->lgSeqs-1] = igap;
      }
    }
    /* Check if we moved. If not, stop */
    if (hasmoved) {
      BiotextPtr->currentC--;
      c--;
    } else {
      break;
    }
  }

  /* cursor goes left */
  BiotextConfigCursor(BiotextPtr);
  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextLockCmd --
 *
 *      This procedure implements the "lock" method.
 *      lock = 1 avoid character deletion/insertion
 *      lock = 0 allows it.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextLockCmd(register Biotext *BiotextPtr,
	       Tcl_Interp *interp,
	       int objc,
	       Tcl_Obj * const objv[]) 
{
  int lock;

  if (objc != 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "lock <boolean>");
    return TCL_ERROR;
  }

  if (Tcl_GetBooleanFromObj(interp, objv[2], &lock) != TCL_OK) 
    return TCL_ERROR;
  BiotextPtr->Lock = lock;

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextInsertCols --
 *
 *      This procedure inserts <count> columns in 
 *      biotext before/after index <i>, depending on the
 *      sign of count (<0 : left, >0 : right)
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextInsertCols(register Biotext *BiotextPtr,
		  int index,
		  int count) 
{
  int i, j, itag, lgs, nseq, stop1, stop2, start;

  lgs  = BiotextPtr->lgSeqs + abs(count);
  nseq = BiotextPtr->nbSeqs;

  /* 
   * First :
   * Reallocate sequence, mapping and tag matrices at the 
   * good length if there is not enough place already
   * allocated
   */
  if (lgs > BiotextPtr->lgAlloc) {
    for (i=0;i<nseq;i++) {
      BiotextPtr->SeqMat[i] = (char *)ckrealloc((char *)BiotextPtr->SeqMat[i],(lgs+1)*sizeof(char));
      BiotextPtr->MapSeq[i]   = (int *)ckrealloc((char *)BiotextPtr->MapSeq[i],lgs*sizeof(int));
      BiotextPtr->TagSeq[i]   = (int *)ckrealloc((char *)BiotextPtr->TagSeq[i],lgs*sizeof(int));
    }
    BiotextPtr->lgAlloc = lgs;
  }
  BiotextPtr->lgSeqs = lgs;

  /*
   * Second :
   * find tag for gap character
   */
  itag = 0;
  for (i=0;i<BiotextPtr->nbMapping;i++) {
    if (BiotextPtr->Mapping[i].smap == '.') {
      itag = i;
      break;
    }
  }

  /* 
   * Third :
   * Do the insertion before or after index
   */
  if (count < 0) {
    /* insert at the left of index */
    /*      v           v 
     * ABCDEF..   .ABCDEF..
     * 01234567   012345678
     */
    count = abs(count);
    stop1 = index + count - 1;
    start = index;
    stop2 = index + count;
  }  else {
    /* insert at the right of index */
    stop1 = index + count;
    start = index + 1;
    stop2 = index + count + 1;
  }

  //fprintf(stderr,"InsertCols: index %d lgs %d stop1 %d start %d stop2 %d\n",index,lgs,stop1,start,stop2);
  //fflush(stderr);
  
  for (j=0;j<nseq;j++) {
    for (i=lgs-1;i>stop1;i--) {
      BiotextPtr->SeqMat[j][i] = BiotextPtr->SeqMat[j][i-count];
      BiotextPtr->MapSeq[j][i]   = BiotextPtr->MapSeq[j][i-count];
      BiotextPtr->TagSeq[j][i]   = BiotextPtr->TagSeq[j][i-count];
    }
  }
  for (j=0;j<nseq;j++) {
    for (i=start;i<stop2;i++) {
      //BiotextPtr->SeqMat[j][i] = '.';
      BiotextPtr->SeqMat[j][i] = 46;
      BiotextPtr->MapSeq[j][i]   = itag;
      BiotextPtr->TagSeq[j][i]   = -1;
    }
  }

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextInsertRows --
 *
 *      This procedure inserts <count> rows in 
 *      biotext before/after index i, depending on the
 *      sign of count.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextInsertRows(register Biotext *BiotextPtr,
		  int index,
		  int count) 
{
  int i, j, nbs, itag, stop1, stop2, start;

  nbs = BiotextPtr->nbSeqs + abs(count);

  BiotextPtr->SeqMat = (char **)ckrealloc((char *)BiotextPtr->SeqMat,nbs*sizeof(char *));
  BiotextPtr->MapSeq = (int **)ckrealloc((char *)BiotextPtr->MapSeq,nbs*sizeof(int *));
  BiotextPtr->TagSeq = (int **)ckrealloc((char *)BiotextPtr->TagSeq,nbs*sizeof(int *));
  BiotextPtr->SeqGrp = (int *)ckrealloc((char *)BiotextPtr->SeqGrp,nbs*sizeof(int));
  /* Allocate the added sequence to the 
   * actual length of the alignment
   */
  for (i=BiotextPtr->nbSeqs;i<nbs;i++) {
    BiotextPtr->SeqMat[i] = (char *)ckalloc((BiotextPtr->lgAlloc+1)*sizeof(char));
    BiotextPtr->MapSeq[i] = (int *)ckalloc(BiotextPtr->lgAlloc*sizeof(int));
    BiotextPtr->TagSeq[i] = (int *)ckalloc(BiotextPtr->lgAlloc*sizeof(int));
  }

  /* mapping index for insertion character */
  itag = 0;
  for (i=0;i<BiotextPtr->nbMapping;i++) {
    if (BiotextPtr->Mapping[i].smap == '.') {
      itag = i;
      break;
    }
  }

  /*
   * if count < 0 insert before index, after otherwise
   */
  if (count < 0) {
    count = abs(count);
    stop1 = index + count - 1;
    start = index;
    stop2 = index + count;
  }  else {
    stop1 = index + count;
    start = index + 1;
    stop2 = index + count + 1;
  }

  for (i=nbs-1;i>stop1;i--) {
    BiotextPtr->SeqGrp[i] = BiotextPtr->SeqGrp[i-count];

    for (j=0; j<BiotextPtr->lgSeqs;j++) {
      BiotextPtr->SeqMat[i][j] = BiotextPtr->SeqMat[i-count][j];
      BiotextPtr->MapSeq[i][j] = BiotextPtr->MapSeq[i-count][j];
      BiotextPtr->TagSeq[i][j] = BiotextPtr->TagSeq[i-count][j];
    }
  }

  for (i=start;i<stop2;i++) {
    BiotextPtr->SeqGrp[i] = 0;
    for (j=0;j<BiotextPtr->lgSeqs;j++) {
      BiotextPtr->SeqMat[i][j] = 46;
      BiotextPtr->MapSeq[i][j] = itag;
      BiotextPtr->TagSeq[i][j] = -1;
    }
  }

  BiotextPtr->nbSeqs = nbs;

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextDeleteCmd --
 *
 *	This procedure is invoked to process the 
 *      "delete" method. It deletes columns of 
 *      gaps or sequences after or before index , 
 *      depending on the sign of count.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextDeleteCmd(register Biotext *BiotextPtr,
		Tcl_Interp *interp,
		int objc,
		Tcl_Obj * const objv[]) 
{
  char *what, *iStr, dummy;
  int i, is, if, result;

  if (objc < 4 && (strcmp(Tcl_GetString(objv[2]),"colsgaps"))) {
    Tcl_WrongNumArgs(interp, 1, objv, " delete <colsgaps|cols|rows> ?index-list?");
    return TCL_ERROR;
  }
  
  /* 
   * First check if we want to remove columns of gaps
   */
  what = Tcl_GetString(objv[2]);
  if (strcasecmp(what,"colsgaps") == 0) {
    result = BiotextDeleteColsGaps(BiotextPtr);
    goto done;
  }

  /* Deletion according to index. Check it */
  iStr = Tcl_GetString(objv[3]);
  if ((strcasecmp(iStr, "all") && strcasecmp(iStr, "end")) && (sscanf(iStr, "%d%s", &i, &dummy) != 1)) {
    Tcl_WrongNumArgs(interp, 1, objv, "<colsgaps|rows|cols> ?all|end|index-list?");
    return TCL_ERROR;
  }
  
  if (strcasecmp(what,"cols") == 0) {
    if (strcasecmp(iStr, "end") == 0) {
      i = BiotextPtr->lgSeqs - 1;
    } else {
      sscanf(iStr, "%d", &i);
    }
    result = BiotextDeleteCols(BiotextPtr, i);
  } else if (strcasecmp(what, "rows") == 0) {
    if (strcasecmp(iStr, "end") == 0) {
      is = BiotextPtr->nbSeqs - 1;
      if = is+1;
    } else if {strcasecmp(iStr, "all") == 0) {
      is = 0;
      if = BiotextPtr->nbSeqs;
    } else {
      sscanf(iStr, "%d", &is);
      if = is+1;
    }
    for (i=is;i<if;i++) {
      result = BiotextDeleteRows(BiotextPtr, i);
    }
  } else {
    // not rows,cols,colsgap so error
    Tcl_WrongNumArgs(interp, 1, objv, "delete <colsgaps|rows|cols> ?index-list?");

    return TCL_ERROR;
  }    
  
 done:
  if (result == TCL_OK) {
    BiotextPtr->flags |= UPDATE_SCROLLBAR;
    BiotextEventuallyRedraw(BiotextPtr);
  }

  return result;
}

/*--------------------------------------------------------------
 *
 * BiotextDeleteColsGaps --
 *
 *      This procedure deletes columns of gaps 
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextDeleteColsGaps(register Biotext *BiotextPtr) 
{
  int i, j, isGap, *colToDel, c;

  colToDel = (int *)ckalloc(BiotextPtr->lgSeqs*sizeof(int));
  for (i=BiotextPtr->lgSeqs-1;i>=0;i--) {
    isGap = 1;
    for (j=0;j<BiotextPtr->nbSeqs;j++) {
      if (BiotextPtr->SeqMat[j][i] != '.') {
	isGap = 0;
	break;
      }
    }
    colToDel[i] = isGap;
  }
  
  c = BiotextPtr->lgSeqs-1;
  for (i=c;i>=0;i--)
    if (colToDel[i])
      BiotextDeleteCols(BiotextPtr, i);
  
  ckfree((char *)colToDel);

  //fprintf(stdout,"\nRemColsGaps : lgs %d currC %d\n",BiotextPtr->lgSeqs, BiotextPtr->currentC);
  //fflush(stdout);

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextDeleteCols --
 *
 *      This procedure deletes <count> columns in 
 *      biotext before/after index i, depending on the
 *      sign of count.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextDeleteCols(register Biotext *BiotextPtr,
		  int index) 
{
  int i, j, lgs, nseq;

  lgs  = BiotextPtr->lgSeqs;
  nseq = BiotextPtr->nbSeqs;

  if (index != lgs-1) {
    for (j=0;j<nseq;j++) {
      for (i=index;i<lgs-1;i++) {
	BiotextPtr->SeqMat[j][i] = BiotextPtr->SeqMat[j][i+1];
	BiotextPtr->MapSeq[j][i] = BiotextPtr->MapSeq[j][i+1];
	if (BiotextPtr->TagSeq != NULL)
	  BiotextPtr->TagSeq[j][i] = BiotextPtr->TagSeq[j][i+1];
      }
      BiotextPtr->SeqMat[j][lgs-1] = '.';
    }
  }

  BiotextPtr->lgSeqs--;
  
  if (BiotextPtr->currentC > BiotextPtr->lgSeqs-1)
    BiotextPtr->currentC = BiotextPtr->lgSeqs - 1;

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextDeleteRows --
 *
 *      This procedure deletes the row in biotext at 
 *      the given index position.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextDeleteRows(register Biotext *BiotextPtr,
		  int index) 
{
  int i, j, nbs;

  nbs = BiotextPtr->nbSeqs;
  for (i=index;i<nbs-1;i++) {
    BiotextPtr->SeqGrp[i] = BiotextPtr->SeqGrp[i+1];
    for (j=0; j<BiotextPtr->lgSeqs; j++) {
      BiotextPtr->SeqMat[i][j] = BiotextPtr->SeqMat[i+1][j];
      BiotextPtr->MapSeq[i][j] = BiotextPtr->MapSeq[i+1][j];
      if (BiotextPtr->TagSeq != NULL)
	BiotextPtr->TagSeq[i][j] = BiotextPtr->TagSeq[i+1][j];
    }
  }
  if (index <= BiotextPtr->currentR)
    BiotextPtr->currentR--;
  
  BiotextPtr->SeqMat = (char **)ckrealloc((char *)BiotextPtr->SeqMat,nbs*sizeof(char *));
  BiotextPtr->MapSeq   = (int **)ckrealloc((char *)BiotextPtr->MapSeq,nbs*sizeof(int *));
  if (BiotextPtr->TagSeq != NULL)
    BiotextPtr->TagSeq   = (int **)ckrealloc((char *)BiotextPtr->TagSeq,nbs*sizeof(int *));

  BiotextPtr->SeqGrp = (int *)ckrealloc((char *)BiotextPtr->SeqGrp, nbs*sizeof(int));

  BiotextPtr->nbSeqs = nbs - 1;

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextGroupCmd --
 *
 *      This routine handles the "group" method.
 *      This routine groups sequences given as a 
 *      list. Subsequent action on any sequence of 
 *      group will affect all sequences of the group.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextGroupCmd(register Biotext *BiotextPtr,
		Tcl_Interp *interp,
		int objc,
		Tcl_Obj * const objv[]) 
{
  char *smode;
  int nseq, i, idx, nmGrp, inm, isq, mode;
  Tcl_Obj **OIdx;

  /*
   * !! GROUP NAME IS HANDLED BY THE TCL CALLER !!!
   *
   * keep following code in case ....
   * Find a group name. Loop through existing names 
   * and find the first missing one
   */

  /*
    isIn = 1;
    nmGrp = 0;
    while (isIn) {
    nmGrp++;
    isIn = 0;
    for(i=0,i<BiotextPtr->nbrGrps;i++) {
    Tcl_ListObjIndex(interp, BiotextPtr->GroupList, i, &el);
    Tcl_GetIntFromObj(interp, el, &val);
    if (nmGrp == val) isIn = 1;
    }
    }
  */

  /*
   * Syntax: <pathname> group ?-new|-merge? name index
   *
   * mode = 1 : 
   * By default, the mode of group creation is "new".
   * If sequences to be grouped already belong to a 
   * present group, they are discared from that 
   * group
   * mode = 0 :
   * Sequences are merged 
   */
  mode = 1;
  if (objc == 4) {
    inm = 2;
    isq = 3;
  } else if (objc == 5) {
    smode = Tcl_GetString(objv[2]);
    if (strcmp(smode, "-new") && strcmp(smode, "-merge")) {
      return TCL_ERROR;
    }
    if (strcmp(smode, "-new") == 0) mode = 1;
    else mode = 0;
    inm = 3;
    isq = 4;
  } else {
    Tcl_WrongNumArgs(interp, 1, objv, "group ?-merge|-new? <group-index> <index-list>");
    return TCL_ERROR;
  }

  Tcl_GetIntFromObj(interp, objv[inm], &nmGrp);
  if (nmGrp == 0) return TCL_OK;
  if (mode) {
    if (BiotextPtr->GroupList == NULL) 
      BiotextPtr->GroupList = Tcl_NewListObj(0, NULL);
    Tcl_ListObjAppendElement(interp, BiotextPtr->GroupList, Tcl_NewIntObj(nmGrp));
    BiotextPtr->nbrGrps++;
    
    Tcl_ListObjGetElements(interp,objv[isq],&nseq,&OIdx);
    for (i=0;i<nseq;i++) {
      Tcl_GetIntFromObj(interp, OIdx[i], &idx);
      BiotextPtr->SeqGrp[idx] = nmGrp;
    }
  } else {
  }

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextUngroupCmd --
 *
 *      This procedure ungroups sequences given as an 
 *      index list. 
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextUngroupCmd(register Biotext *BiotextPtr,
		  Tcl_Interp *interp,
		  int objc,
		  Tcl_Obj *const objv[]) 
{
  int nseq, i, j, idx, val, isIn;
  Tcl_Obj **OIdx, *el;

  if (objc != 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "ungroup <index-list>");
    return TCL_ERROR;
  }
  
  /*
   * !! GROUPE NAME GERE PAR LA COUCHE TCL !!!
   *
   * Find a group name. Loop through existing names 
   * and find the first missing one
   */

  /*
    isIn = 1;
    nmGrp = 0;
    while (isIn) {
    nmGrp++;
    isIn = 0;
    for(i=0,i<BiotextPtr->nbrGrps;i++) {
    Tcl_ListObjIndex(interp, BiotextPtr->GroupList, i, &el);
    Tcl_GetIntFromObj(interp, el, &val);
    if (nmGrp == val) isIn = 1;
    }
    }
  */

  Tcl_ListObjGetElements(interp,objv[2],&nseq,&OIdx);
  for (i=0;i<nseq;i++) {
    Tcl_GetIntFromObj(interp, OIdx[i], &idx);
    BiotextPtr->SeqGrp[idx] = 0;
  }

  /*
   * Check if a group is empty, or contains only one 
   * sequence. Beware, Sequences may be in the buffer.
   */
  for(i=0;i<BiotextPtr->nbrGrps;i++) {
    Tcl_ListObjIndex(interp, BiotextPtr->GroupList, i, &el);
    Tcl_GetIntFromObj(interp, el, &val);
    isIn = 0;
    for (j=0;j<BiotextPtr->nbSeqs;j++) {
      if (BiotextPtr->SeqGrp[j] == val) {
	isIn++;
      }
    }
    if (isIn < 2 && BiotextPtr->nbrSeqsCache > 0) {
      for(j=0;j<BiotextPtr->nbrSeqsCache;j++) {
	if (BiotextPtr->SeqGrpCache[j] == val) {
	  isIn++;
	}
      }
    }
    if (isIn < 2) 
      BiotextDeleteGroup(BiotextPtr, val);
  }

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextCursorCmd --
 *
 *      This procedure implements the "cursor" mathod.
 *      It sets or gives the cursor position or 
 *      set it to the given index.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextCursorCmd(register Biotext *BiotextPtr,
		 Tcl_Interp *interp,
		 int objc,
		 Tcl_Obj * const objv[]) 
{
  char *index = NULL;
  char bufr[INDEX_BUFSIZE], bufc[INDEX_BUFSIZE];
  int r, c;
  
  if (objc > 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "index  ?index?");
    return TCL_ERROR;
  }
  
  if (objc == 3) {
    index = Tcl_GetString(objv[2]);
    BiotextGetIndex(BiotextPtr, index, &r, &c);

    /* BiotextGetIndex gives indexes in the biotext context */
    BiotextPtr->currentC = c;
    BiotextPtr->currentR = r;
  }

  /* for output,
     add 1 to row index as it starts at 1
  */
  sprintf(bufr,"%d",BiotextPtr->currentR+1);
  sprintf(bufc,"%d",BiotextPtr->currentC);
  
  Tcl_SetResult(interp, bufr, NULL);
  Tcl_AppendResult(interp,".",bufc, (char *)NULL);
  
  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextIndexCmd --
 *
 *      This procedure implements the "index" method.
 *      It gives the row and col coordinates 
 *      equivalent to the index given as argument.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextIndexCmd(register Biotext *BiotextPtr,
		Tcl_Interp *interp,
		int objc,
		Tcl_Obj * const objv[]) 
{
  char *index = NULL;
  char bufr[INDEX_BUFSIZE], bufc[INDEX_BUFSIZE];
  int r, c;
  
  if (objc != 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "index  ?index?");
    return TCL_ERROR;
  }
  
  index = Tcl_GetString(objv[2]);
  BiotextGetIndex(BiotextPtr, index, &r, &c);
  
  /* for output,
     add 1 to row index as it starts at 1
  */
  sprintf(bufr,"%d",r+1);
  sprintf(bufc,"%d",c);
  
  Tcl_SetResult(interp, bufr, NULL);
  Tcl_AppendResult(interp,".",bufc, (char *)NULL);
  
  return TCL_OK;
}


/*--------------------------------------------------------------
 *
 * BiotextPushCmd --
 *
 *      This procedure handles the "push" method.
 *      The sequence(s) at current index are pushed 
 *      to the left or right up to the next gap.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextPushCmd(register Biotext *BiotextPtr,
	       Tcl_Interp *interp,
	       int objc,
	       Tcl_Obj * const objv[]) 
{
  int r, c, left, nextGap, i, icol, nGrp, col;
  char *what;

  if (objc < 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "pushleft <index>");
    return TCL_ERROR;
  }

  left = 0;
  what = Tcl_GetString(objv[2]);
  if (strcmp(what,"left") && strcmp(what, "right")) {
    return TCL_ERROR;
  }
  if (strcmp(what, "left") == 0) 
    left = 1;

  /*
   * Get index :
   * <r> is the index of the reference sequence
   */
  r = BiotextPtr->currentR;
  c = BiotextPtr->currentC;
  icol = c;

  //fprintf(stdout,"Push : r= %3d, c= %3d\n",r,c);
  //fflush(stdout);

  /* 
   * Find the closest gap to the left/right
   */
  if (left) {
    col = -c;
  } else {
    col = c;
  }

  /* Is the sequence in a group ? */
  nGrp = BiotextPtr->SeqGrp[r];
  if (nGrp != 0 && BiotextPtr->nbrGrps > 0) {
    nextGap = BiotextFindNextGapOfGroup(BiotextPtr, r, nGrp, col);
  } else {
    nextGap = BiotextFindNextGap(BiotextPtr, r, col);
  }

  /*
   * if nextGap == -1/end, we are at the beginning/end 
   * of the alignment, so insert a gap column before 
   * col 0 or after last col.
   * Remove the gap at nextGap, push sequences
   * and insert gap at c.
   */
  if (nextGap == -1) {
    BiotextInsertCols(BiotextPtr, 0, -1);
    nextGap = 0;

    /* 
     * If we insert cols to the left, the "current"
     * and icol positions should be increased by 1 
     */
    BiotextPtr->currentC++;
    //BiotextPtr->leftIndex++;
    icol++;

    /* we have to update the scrollbars */
    BiotextPtr->flags |= UPDATE_SCROLLBAR;
  } else if (nextGap == BiotextPtr->lgSeqs) {
    //fprintf(stdout,"--> lgs AV Ins %d\n",BiotextPtr->lgSeqs);
    //fflush(stdout);
    BiotextInsertCols(BiotextPtr, BiotextPtr->lgSeqs-1, 1);
    nextGap = BiotextPtr->lgSeqs-1;
    //fprintf(stdout,"--> lgs AP ins %d\n",BiotextPtr->lgSeqs);
    //fflush(stdout);

    /* we have to update the scrollbars */
    BiotextPtr->flags |= UPDATE_SCROLLBAR;
  }

  /* 
   * We can then push the sequence
   */
  //fprintf(stderr,"av do Push : c=%3d lgs= %3d\n",BiotextPtr->currentC, BiotextPtr->lgSeqs);
  //fflush(stderr);
  if (nGrp != 0 && BiotextPtr->nbrGrps > 0) {
    for(i=0;i<BiotextPtr->nbSeqs;i++) 
      if (BiotextPtr->SeqGrp[i] == nGrp) 
	BiotextPushInSeq(BiotextPtr, i, icol, nextGap);
  } else {
    BiotextPushInSeq(BiotextPtr, r, icol, nextGap);
  }

  if (left) 
    BiotextPtr->currentC--;
  else 
    BiotextPtr->currentC++;

  //fprintf(stderr,"fin Push : c=%3d lgs= %3d\n",BiotextPtr->currentC, BiotextPtr->lgSeqs);
  //fflush(stderr);

  //BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);

  //fprintf(stderr,"fin Push : c=%3d lgs= %3d\n\n",BiotextPtr->currentC, BiotextPtr->lgSeqs);
  //fflush(stderr);

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextSeeCmd --
 *
 *      This procedure handles the "see" method.
 *      The biotext widget adjusts the view in order 
 *      to make the index position visible.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextSeeCmd(register Biotext *BiotextPtr,
	      Tcl_Interp *interp,
	      int objc,
	      Tcl_Obj * const objv[]) 
{
  char *index;
  char firstStr[TCL_DOUBLE_SPACE+1], lastStr[TCL_DOUBLE_SPACE+1];
  int result, row, col, firstC, lastC, firstR, lastR;
  int width, height, firstX, lastX, firstY, lastY;
  int visible, same, totalX, totalY;
  double min, max;

  if (objc != 3) {
    Tcl_WrongNumArgs(interp, 2, objv, "index");
    return TCL_ERROR;
  }
  index = Tcl_GetString(objv[2]);
  BiotextGetIndex(BiotextPtr, index, &row, &col);

  /* boundaries */
  width  = BiotextPtr->widthChar;
  height = BiotextPtr->heightChar;
  row = MAX(0,row);
  row = MIN(row,BiotextPtr->nbSeqs-1);
  col = MAX(0,col);
  col = MIN(col,BiotextPtr->lgSeqs-1);

  /* if index is already visible, return */
  visible = Biotext_IsVisible(BiotextPtr, col, row, 1);
  same    = (row == BiotextPtr->currentR) && (col == BiotextPtr->currentC);
  totalX = BiotextPtr->lgSeqs*BiotextPtr->charWidth + 2*(BiotextPtr->borderWidth + BiotextPtr->inset);
  totalY = BiotextPtr->nbSeqs*BiotextPtr->charHeight + 2*(BiotextPtr->borderWidth + BiotextPtr->inset);

  if (! visible) {
    if (same || (row != BiotextPtr->currentR)) {
      /* Update only y-scrollbar */
      firstC = col;
      lastC  = col;
      if ((row-height/2)<0) {
	firstR = 0;
	lastR  = height - 1;
	min = 0;
	Biotext_Coords2Pixels(BiotextPtr, lastC, lastR, &lastX, &lastY, 0);
	max = (double)lastY/totalY;
      } else if (row>BiotextPtr->nbSeqs-1-height/2) {
	firstR = BiotextPtr->nbSeqs - height;
	lastR  = BiotextPtr->nbSeqs - 1;
	max = 1.0;
	Biotext_Coords2Pixels(BiotextPtr, firstC, firstR, &firstX, &firstY, 0);
	min = (double)firstY/totalY;
      } else {
	firstR = row-height/2;
	lastR  = firstR + height - 1;
	Biotext_Coords2Pixels(BiotextPtr, firstC, firstR, &firstX, &firstY, 0);
	min = (double)firstY/totalY;
	Biotext_Coords2Pixels(BiotextPtr, lastC, lastR, &lastX, &lastY, 0);
	max = (double)lastY/totalY;
      }
      
      /*
       * Send X and Y first and last position to the
       * scrollbars.
       */
      Tcl_Preserve(interp);
      
      /* Y - scrollbar */
      firstStr[0] = lastStr[0] = ' ';
      Tcl_PrintDouble(NULL, min, firstStr+1);
      Tcl_PrintDouble(NULL, max, lastStr+1);
      
      result = Tcl_VarEval(interp, BiotextPtr->yscrollCmd, firstStr, lastStr, NULL);
      if (result != TCL_OK) {
	Tcl_AddErrorInfo(interp, "\n    (vertical scrolling command executed by biotext)");
	Tcl_BackgroundError(interp);
      }
      
      BiotextPtr->topIndex  = firstR;
      BiotextPtr->currentR = row;
      Tcl_Release(interp);
    }
        
    if (same || (col != BiotextPtr->currentC)) {
      /* Update only x-scrollbar */
      firstR = row;
      lastR  = row;
      if (col-width/2 < 0) {
	firstC = 0;
	lastC  = width - 1;
	min = 0.0;
	Biotext_Coords2Pixels(BiotextPtr, lastC, lastR, &lastX, &lastY, 0);
	max = (double)lastX/totalX;
      } else if (col > BiotextPtr->lgSeqs-1-width/2) {
	firstC = BiotextPtr->lgSeqs - width;
	lastC  = BiotextPtr->lgSeqs - 1;
	max = 1.0;
	Biotext_Coords2Pixels(BiotextPtr, firstC, firstR, &firstX, &firstY, 0);
	min = (double)firstX/totalX;
      } else {
	firstC = col-width/2;
	lastC  = firstC + width;
	Biotext_Coords2Pixels(BiotextPtr, firstC, firstR, &firstX, &firstY, 0);
	min = (double)firstX/totalX;
	Biotext_Coords2Pixels(BiotextPtr, lastC, lastR, &lastX, &lastY, 0);
	max = (double)lastX/totalX;
      }

      BiotextPtr->leftIndex = firstC;
      BiotextPtr->currentC = col;
      
      /*
       * Send X and Y first and last fractional position to the
       * scrollbars.
       */
      Tcl_Preserve(interp);
      
      /* X - scrollbar */
      firstStr[0] = lastStr[0] = ' ';
      Tcl_PrintDouble(NULL, min, firstStr+1);
      Tcl_PrintDouble(NULL, max, lastStr+1);
      
      result = Tcl_VarEval(interp, BiotextPtr->xscrollCmd, firstStr, lastStr, NULL);
      if (result != TCL_OK) {
	Tcl_AddErrorInfo(interp, "\n    (horizontal scrolling command executed by biotext)");
	Tcl_BackgroundError(interp);
      }
      
      Tcl_Release(interp);
    }

    BiotextPtr->flags |= UPDATE_SCROLLBAR;
  } else {
    BiotextPtr->currentC = col;
    BiotextPtr->currentR = row;
  }

  BiotextConfigCursor(BiotextPtr);
  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextCopyCmd --
 *
 *      This procedure handles the "copy" method.
 *      The sequences corresponding to the indices
 *      given are copied to the cut/copy/paste buffer.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextCopyCmd(register Biotext *BiotextPtr,
	       Tcl_Interp *interp,
	       int objc,
	       Tcl_Obj * const objv[]) 
{
  int nseq;
  Tcl_Obj **OIdx, *evalObj, *resObj;

  if (objc != 3) {
    Tcl_WrongNumArgs(interp, 2, objv, " copy <index-list>");
    return TCL_ERROR;
  }

  Tcl_ListObjLength(interp,objv[2],&nseq);
  if (nseq == 0) return TCL_OK;

  /*
   * sort the index list in decreasing order
   */
  evalObj = Tcl_NewListObj(0, NULL);
  Tcl_ListObjAppendElement(NULL, evalObj, Tcl_NewStringObj("lsort", -1));
  Tcl_ListObjAppendElement(NULL, evalObj, Tcl_NewStringObj("-integer", -1));
  Tcl_ListObjAppendElement(NULL, evalObj, Tcl_NewStringObj("-decreasing", -1));
  Tcl_ListObjAppendElement(NULL, evalObj, objv[2]);

  if (Tcl_EvalObjEx(interp, evalObj, TCL_EVAL_DIRECT) != TCL_OK) {
    return TCL_ERROR;
  }

  resObj = Tcl_NewListObj(0, NULL);
  resObj = Tcl_GetObjResult(interp);
  Tcl_ListObjGetElements(interp, resObj,&nseq, &OIdx);
  //Tcl_ResetResult(interp);

  /*
   * Initialize the copy buffer. Only one step back
   * copy/paste allowed at present
   */
  BiotextInitBuffer(BiotextPtr, nseq);
  BiotextCopySeqsToBuffer(BiotextPtr, nseq, OIdx);

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextCutCmd --
 *
 *      This procedure handles the "cut" method.
 *      The sequences corresponding to the indices
 *      given are cut from the widget, and stored in
 *      memory.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextCutCmd(register Biotext *BiotextPtr,
	      Tcl_Interp *interp,
	      int objc,
	      Tcl_Obj * const objv[]) 
{
  int nseq, i, idx;
  Tcl_Obj **OIdx, *resObj, *evalObj;

  if (objc != 3) {
    Tcl_WrongNumArgs(interp, 2, objv, "<index-list>");
    return TCL_ERROR;
  }

  Tcl_ListObjLength(interp,objv[2],&nseq);
  if (nseq == 0) return TCL_OK;

  /*
   * sort the index list in decreasing order
   */
  evalObj = Tcl_NewListObj(0, NULL);
  Tcl_ListObjAppendElement(NULL, evalObj, Tcl_NewStringObj("lsort", -1));
  Tcl_ListObjAppendElement(NULL, evalObj, Tcl_NewStringObj("-integer", -1));
  Tcl_ListObjAppendElement(NULL, evalObj, Tcl_NewStringObj("-decreasing", -1));
  Tcl_ListObjAppendElement(NULL, evalObj, objv[2]);

  if (Tcl_EvalObjEx(interp, evalObj, TCL_EVAL_DIRECT) != TCL_OK) {
    return TCL_ERROR;
  }

  resObj = Tcl_NewListObj(0, NULL);
  resObj = Tcl_GetObjResult(interp);
  Tcl_ListObjGetElements(interp, resObj,&nseq, &OIdx);

  /*
   * Initialize the copy buffer. 
   * Only one step back
   * copy/paste allowed at present
   */
  BiotextInitBuffer(BiotextPtr, nseq);
  BiotextCopySeqsToBuffer(BiotextPtr, nseq, OIdx);

  /*
   * Now delete the sequences 
   */
  for (i=0; i<nseq; i++) {
    Tcl_GetIntFromObj(interp, OIdx[i], &idx);
    BiotextDeleteRows(BiotextPtr, idx);
  }

  /* redraw the widget */
  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextPasteCmd --
 *
 *      This procedure handles the "paste" method.
 *      The sequences in the paste buffer are copied 
 *      back inside the widget after the given index.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextPasteCmd(register Biotext *BiotextPtr,
		Tcl_Interp *interp,
		int objc,
		Tcl_Obj * const objv[]) 
{
  char *res;
  int idx, i, j, k, diff;

  if (objc != 3) {
    Tcl_WrongNumArgs(interp, 2, objv, "index");
    return TCL_ERROR;
  }

  /*
   * If nothing to paste, return
   */
  if (BiotextPtr->nbrSeqsCache == 0)
    return TCL_OK;

  /*
   * Get the insertion point.
   */
  res = Tcl_GetString(objv[2]);
  if (strcasecmp(res, "end") == 0) 
    idx = BiotextPtr->nbSeqs-1;
  else {
    if (Tcl_GetInt(interp, res, &idx) != TCL_OK) 
      return TCL_ERROR;
  }

  /*
   * First insert empty rows ...
   */
  BiotextInsertRows(BiotextPtr, idx, BiotextPtr->nbrSeqsCache);

  /* Check if length of sequences to be inserted is 
     the same of the one already present
  */
  diff = BiotextPtr->lgSeqs - BiotextPtr->lgSeqsCache;
  if (diff > 0) {
    /* we must add cols to buffer seqs */
    BiotextAddColsToBufferSeqs(BiotextPtr,diff);
  } else if (diff < 0) {
    /* we must add cols to present seqs */
    diff = abs(diff);
    BiotextInsertCols(BiotextPtr, BiotextPtr->lgSeqs-1, diff);
  }
  
  /* ... and then fill them */
  for (k=BiotextPtr->nbrSeqsCache-1,i=idx+1; k>=0;i++,k--) {
    BiotextPtr->SeqGrp[i] = BiotextPtr->SeqGrpCache[k];
    for (j=0;j<BiotextPtr->lgSeqs;j++) {
      BiotextPtr->SeqMat[i][j] = BiotextPtr->SeqMatCache[k][j];
      BiotextPtr->MapSeq[i][j] = BiotextPtr->MapSeqCache[k][j];
      BiotextPtr->TagSeq[i][j] = BiotextPtr->TagSeqCache[k][j];
    }
  }

  /* Indicate there's nothing more to paste */
  BiotextInitBuffer(BiotextPtr,0);

  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextFontCmd --
 *
 *      This procedure handles the "font" method.
 *      This routine sets a new font, or increase or
 *      decrease the size of the current font.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextFontCmd(register Biotext *BiotextPtr,
	       Tcl_Interp *interp,
	       int objc,
	       Tcl_Obj * const objv[]) 
{
  char *what;

  if (objc != 3 && objc != 4 && objc != 6 && objc != 8) {
    Tcl_WrongNumArgs(interp, 2, objv, "?family <family>? ?size <size>? ?weight <weight>? | <bigger|smaller>");
    return TCL_ERROR;
  }

  static CONST84 char *fontCmdNames[] = {
    "family", "size", "weight", (char *)NULL
  };
  enum fontCommand {
    CMD_FFAMILY, CMD_FSIZE, CMD_FWEIGHT
  };

  if (objc == 3) {
    what = Tcl_GetString(objv[2]);
    if (strcmp(what, "bigger") == 0) 
      BiotextPtr->FontSize++;
    if (strcmp(what, "smaller") == 0) {
      if (BiotextPtr->FontSize > 4) 
	BiotextPtr->FontSize--;
    }
    if (strcmp(what, "XLFD") == 0) {
      Tcl_SetObjResult(interp, Tcl_NewStringObj(BiotextPtr->FontXlfd, -1));
      return TCL_OK;
    }

    goto done;
  }

  int cmdIndex, i, size, result;
  char *fam, *wgt;

  for (i=2;i<objc;i++) {
    if (Tcl_GetIndexFromObj(interp, objv[i], fontCmdNames, "font option", 0, &cmdIndex) != TCL_OK) {
      return TCL_ERROR;
    }
    
    switch ((enum fontCommand) cmdIndex) {
    case CMD_FFAMILY:
      i++;
      fam = Tcl_GetString(objv[i]);
      strcpy(BiotextPtr->FontFamily,fam);
      break;
    case CMD_FSIZE:
      i++;
      what = Tcl_GetString(objv[i]);
      result = Tcl_GetInt(interp, what, &size);
      if (result != TCL_OK) {
	Tcl_AddErrorInfo(interp, ": Font size must be an interger");
	Tcl_BackgroundError(interp);
      } else {
	if (size < 4) size = 4;
	BiotextPtr->FontSize = size;
      }
      break;
    case CMD_FWEIGHT:
      i++;
      wgt = Tcl_GetString(objv[i]);
      Tcl_UtfToLower(wgt);
      if (strcmp(wgt, "normal") && strcmp(wgt, "bold")) {
	Tcl_AddErrorInfo(interp, " : Font weight should be normal or bold");
	Tcl_BackgroundError(interp);
      } else {
	strcpy(BiotextPtr->FontWeight,wgt);
      }
    }
  }
  
 done:
  BiotextSetFont(BiotextPtr);

  BiotextPtr->flags |= UPDATE_SCROLLBAR;
  BiotextEventuallyRedraw(BiotextPtr);

  return TCL_OK;
}

/*--------------------------------------------------------------
 *
 * BiotextOutputCmd --
 *
 *      This procedure handles the "output" method.
 *      The sequences corresponding to the indices
 *      given are send as output result.
 *
 * Results:
 *	A Tcl object containing a list of sequences
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
int 
BiotextOutputCmd(register Biotext *BiotextPtr,
		 Tcl_Interp *interp,
		 int objc,
		 Tcl_Obj * const objv[]) 
{
  char *idxStr;
  int nseq, i, nelt, idx, lgt;
  Tcl_Obj **OIdx, *resObj, *idxObj, *Oelt;

  if (objc != 3) {
    Tcl_WrongNumArgs(interp, 2, objv, "<index-list>");
    return TCL_ERROR;
  }

  idxObj = Tcl_NewListObj(0, NULL);
  idxStr = Tcl_GetString(objv[2]);
  if (strcmp(idxStr, "all") == 0) {
    nelt = BiotextPtr->nbSeqs;
    for (i=0; i<nelt; i++)
      Tcl_ListObjAppendElement(interp, idxObj, Tcl_NewIntObj(i));
  } else {
    if (Tcl_ListObjGetElements(interp,objv[2],&nseq,&OIdx) != TCL_OK)
      return TCL_ERROR;
    if (nseq == 0) return TCL_OK;
    nelt = nseq;
    for (i=0; i<nseq;i++) 
      Tcl_ListObjAppendElement(interp, idxObj, OIdx[i]);
  }
  
  lgt = BiotextPtr->lgSeqs;
  resObj = Tcl_NewListObj(0, NULL);
  for(i=0; i<nelt; i++) {
    Tcl_ListObjIndex(interp, idxObj, i, &Oelt);
    Tcl_GetIntFromObj(interp, Oelt, &idx);
    Tcl_ListObjAppendElement(interp, resObj, Tcl_NewStringObj(BiotextPtr->SeqMat[idx], lgt));
  }
  
  Tcl_SetObjResult(interp, resObj);

  return TCL_OK;
}


/*--------------------------------------------------------------
 *
 * BiotextCleanCmd --
 *
 *      This procedure handles the "clean" method.
 *      This commands desallocate all sequences, all 
 *      mapping, all tags definition and ranges. 
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */
void  
BiotextCleanCmd(register Biotext *BiotextPtr,
		 Tcl_Interp *interp,
		 int objc,
		 Tcl_Obj * const objv[]) 
{
  int i;

  /* desallocate the sequences, mapping and tag matrix */
  for (i=0;i<BiotextPtr->nbSeqs;i++) {
    ckfree((char *)BiotextPtr->SeqMat[i]);
    ckfree((char *)BiotextPtr->TagSeq[i]);
    ckfree((char *)BiotextPtr->MapSeq[i]);
  }
  ckfree((char *)BiotextPtr->SeqMat);
  ckfree((char *)BiotextPtr->TagSeq);
  ckfree((char *)BiotextPtr->MapSeq);
  BiotextPtr->SeqMat = NULL;
  BiotextPtr->TagSeq = NULL;
  BiotextPtr->MapSeq = NULL;
  /* re-initialize counters */
  BiotextPtr->nbSeqs = -1;
  BiotextPtr->nbMapping = 0;
  BiotextPtr->nbTags = 0;

  /* desallocate groups */
  ckfree((char *)BiotextPtr->SeqGrp);
  /* desallocate tags */
  ckfree((char *)BiotextPtr->Tags);
  BiotextPtr->Tags = NULL;
  /* desallocate mapping */
  ckfree((char *)BiotextPtr->Mapping);
  BiotextPtr->Mapping = NULL;

  /* desallocate cache buffers */
  if (BiotextPtr->nbrSeqsCache > 0) {
    ckfree((char *)BiotextPtr->SeqGrpCache);
    for (i=0; i<BiotextPtr->nbrSeqsCache; i++) {
      ckfree((char *)BiotextPtr->SeqMatCache[i]);
      ckfree((char *)BiotextPtr->MapSeqCache[i]);
      ckfree((char *)BiotextPtr->TagSeqCache[i]);
    }
    ckfree((char *)BiotextPtr->SeqMatCache);
    ckfree((char *)BiotextPtr->MapSeqCache);
    ckfree((char *)BiotextPtr->TagSeqCache);
  }
  BiotextPtr->nbrSeqsCache = 0;

  return;
}






