# biotext.tcl --
#
# This file defines the default bindings for Tk 
# biotext widgets and provides
# procedures that help in implementing those bindings.
#
# Copyright (c) 2010 Luc Moulinier and LBGI,
# (Laboratoire de Biologie et Genomique Integratines)
#


#-------------------------------------------------------------------------
# Elements of tk::Priv that are used in this file:
#
# bufferNTimes - 
# records numeric keys that will, when
# invoked, give how many times a given
# command should be done.
#
# mouseMoved - 
# Non-zero means the mouse has moved a significant
# amount since the button went down (so, for example,
# start dragging out a selection).
#
# pressX - 
# X-coordinate at which the mouse button was pressed.
#
# selectMode -
# The style of selection currently underway:
# char, word, or line.
# x, y -
# Last known mouse coordinates for scanning
# and auto-scanning.
#
# data -
# Used for Cut and Copy
#-------------------------------------------------------------------------
#
# Initialisation -
#

namespace eval ::tk::biotext {
    # Ensure that a namespace is created for us
    variable Priv
    array set Priv [list x 0 y 0 bufN "" mouseMoved 0]
    
    namespace export cursor
}

# Bindings of numbers.
package require platform
set ckp [list KP_Insert KP_End KP_Down KP_Next KP_Left KP_Begin KP_Right KP_Home KP_Up KP_Prior]
set Lc [list 0 1 2 3 4 5 6 7 8 9]
foreach c $Lc k $ckp {
    bind Biotext "<Key-$c>"    {::tk::biotext::bufferNTimes "add" %A}
    bind Biotext "<Key-KP_$c>" {::tk::biotext::bufferNTimes "add" %A}
    if {$::tcl_platform(platform) ne "windows" && ! [regexp {^macos} [::platform::generic]] } {
	bind Biotext <Key-$k>  {::tk::biotext::bufferNTimes "add" %A}
    }
}


bind Biotext <1> {
    tk::biotext::button1 %W %x %y
}
bind Biotext <Shift-Left> {
    %W push left
}
bind Biotext <Shift-Right> {
    %W push right
}

bind Biotext <Left>  {tk::biotext::cursor %W -1  0}
bind Biotext <Right> {tk::biotext::cursor %W  1  0}
bind Biotext <Up>    {tk::biotext::cursor %W  0 -1}
bind Biotext <Down>  {tk::biotext::cursor %W  0  1}

bind Biotext <BackSpace> {
    set ix [%W cursor]
    set n [::tk::biotext::bufferNTimes "empty"]
    %W delchars -group $ix $n
}
bind Biotext <Shift-BackSpace> {
    set ix [%W cursor]
    set n [::tk::biotext::bufferNTimes "empty"]
    %W delchars -group $ix $n
}


#-------------------------------------------------------------------------
# The code below creates the default class  
# bindings for biotext widgets.
#-------------------------------------------------------------------------


# ::tk::biotext::cursor --
# This procedure is invoked to set the cursor place in
# biotext widgets.
# It moves the cursor from the "current" position to
# the desired place taking into
# account the N Times buffer, which default is 1 time.
#
# Arguments:
# w - The biotext window in which to set the cursor
# x - The increment in rows
# y - The increment in columns
proc ::tk::biotext::cursor {w x y} {
    lassign [split [$w cursor] ,] r c
    set n [::tk::biotext::bufferNTimes "empty"]
    $w see [incr r [expr {$y*$n}]],[incr c [expr {$x*$n}]]

    return
}

# ::tk::biotext::button1 --
# This procedure is invoked to handle button-1 
# presses in biotext widgets.
# It moves the insertion cursor and claims the input 
# focus.
#
# Arguments:
# w - The biotext window in which the button was 
# pressed.
# x - The x-coordinate (pixels) of the button press.
# y - The y-coordinate (pixels) of the button press.

proc ::tk::biotext::button1 {w x y} {
    variable Priv
    set Priv(selectMode) char
    set Priv(mouseMoved) 0
    set Priv(pressX) $x
    set Priv(pressY) $y
    $w cursor @$x,$y
    if {"disabled" ne [$w cget -state]} {
	focus $w
    }

    return
}

proc ::tk::biotext::bufferNTimes {what {n ""}} {
    variable Priv

    set res 1
    switch $what {
	"add" {append Priv(bufN) $n}
	"empty" {
	    if {$Priv(bufN) ne ""} {
		set res $Priv(bufN)
		set Priv(bufN) ""
	    }
	}
    }

    return $res
}


# ::tk::BiotextMouseSelect --
# This procedure is invoked when dragging out a selection with
# the mouse.  Depending on the selection mode (character, word,
# line) it selects in different-sized units.  This procedure
# ignores mouse motions initially until the mouse has moved from
# one character to another or until there have been multiple clicks.
#
# Arguments:
# w -		The entry window in which the button was pressed.
# x -		The x-coordinate of the mouse.

proc ::tk::BiotextMouseSelect {w x y} {
    variable ::tk::Priv

    if {(abs($Priv(pressX) - $x) >= 3) || (abs($Priv(pressY) - $y) >= 3)} {
	set Priv(mouseMoved) 1
    }
    switch $Priv(selectMode) {
	char {
	    if {$Priv(mouseMoved)} {
		if {$cur < $anchor} {
		    $w selection range $cur $anchor
		} elseif {$cur > $anchor} {
		    $w selection range $anchor $cur
		} else {
		    $w selection clear
		}
	    }
	}
	word {
	    if {$cur < [$w index anchor]} {
		set before [tcl_wordBreakBefore [$w get] $cur]
		set after [tcl_wordBreakAfter [$w get] [expr {$anchor-1}]]
	    } else {
		set before [tcl_wordBreakBefore [$w get] $anchor]
		set after [tcl_wordBreakAfter [$w get] [expr {$cur - 1}]]
	    }
	    if {$before < 0} {
		set before 0
	    }
	    if {$after < 0} {
		set after end
	    }
	    $w selection range $before $after
	}
	line {
	    $w selection range 0 end
	}
    }
    if {$Priv(mouseMoved)} {
        $w icursor $cur
    }
    update idletasks
}

