/*
* tkBiotext.c --
*
*	This module implements "Biotext" widgets that are object based. A
*	"Biotext" is a widget that displays a single Biotext that can be moved
*	around and resized. This file is intended as an example of how to
*	build a widget; it isn't included in the normal wish, but it is
*	included in "tktest".
*
* Copyright (c) 1997 Sun Microsystems, Inc.
*
* See the file "license.terms" for information on usage and redistribution of
* this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
* RCS: @(#) $Id: tkBiotext.c,v 1.13 2008/10/17 23:18:37 nijtmans Exp $
*/

#include "tkBiotext.h"

#ifndef DLLEXPORT
#define DLLEXPORT __declspec(dllexport)
#endif 

#ifndef DEBUG
#define DEBUG 1
#endif

/*
 * Information used for argv parsing.
 */

static CONST84 Tk_OptionSpec optionSpecs[] = {
  {TK_OPTION_BORDER, "-background", "background", "Background", "#d9d9d9", -1, Tk_Offset(Biotext, bgBorder), 0, (ClientData) "#d9d9d9"},
  {TK_OPTION_SYNONYM, "-bg", NULL, NULL, NULL, 0, -1, 0, (ClientData) "-background"},
  {TK_OPTION_SYNONYM, "-bd", NULL, NULL, NULL, 0, -1, 0, (ClientData) "-borderwidth"},
  {TK_OPTION_PIXELS, "-borderwidth", "borderWidth", "BorderWidth", "2", -1, Tk_Offset(Biotext, borderWidth), 0},
  {TK_OPTION_BOOLEAN, "-dbl", "doubleBuffer", "DoubleBuffer", "1", Tk_Offset(Biotext, doubleBufferPtr), -1},
  {TK_OPTION_SYNONYM, "-fg", NULL, NULL, NULL, 0, -1, 0, (ClientData) "-foreground"},
  {TK_OPTION_BORDER, "-foreground", "foreground", "Foreground", "#b03060", -1, Tk_Offset(Biotext, fgBorder), 0, (ClientData) "white"},
  {TK_OPTION_RELIEF, "-relief", "relief", "Relief", "raised", Tk_Offset(Biotext, reliefPtr), -1},
  {TK_OPTION_INT, "-width", "width", "Width", "60", -1, Tk_Offset(Biotext, widthChar), 0, 0, 0},
  {TK_OPTION_INT, "-height", "height", "Height", "30", -1, Tk_Offset(Biotext, heightChar), 0, 0, 0},    
  {TK_OPTION_STRING, "-xscrollcommand", "xScrollCommand", "ScrollCommand","", -1, Tk_Offset(Biotext, xscrollCmd),TK_OPTION_NULL_OK, 0, 0},
  {TK_OPTION_STRING, "-yscrollcommand", "yScrollCommand", "ScrollCommand","", -1, Tk_Offset(Biotext, yscrollCmd),TK_OPTION_NULL_OK, 0, 0},
  {TK_OPTION_STRING_TABLE, "-state", "state", "State",
 "disabled", -1, Tk_Offset(Biotext, state), 0, (ClientData) stateStrings, 0},
  {TK_OPTION_COLOR, "-cursorcolor", NULL, NULL, "yellow", -1, Tk_Offset(Biotext, cursorColor), TK_OPTION_NULL_OK, 0, 0},
  {TK_OPTION_STRING, "-class", "class", "Class", "Biotext", -1, Tk_Offset(Biotext, className), 0, 0, 0},
  {TK_OPTION_INT, "-insertofftime", "insertOffTime", "OffTime", "300", -1, Tk_Offset(Biotext, insertOffTime), 0, 0, 0},
  {TK_OPTION_INT, "-insertontime", "insertOnTime", "OnTime", "600", -1, Tk_Offset(Biotext, insertOnTime), 0, 0, 0},
  {TK_OPTION_END}
};


/*
 *--------------------------------------------------------------
 *
 * BiotextObjCmd --
 *
 *	This procedure is invoked to process the 
 *      "Biotext" Tcl command. It
 *	creates a new "Biotext" widget.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	A new widget is created and configured.
 *
 *--------------------------------------------------------------
 */

int
BiotextObjCmd(ClientData clientData,
	      Tcl_Interp *interp,
	      int objc,
	      Tcl_Obj *const objv[]) 
{
  register Biotext *BiotextPtr;
  Tk_Window tkwin;
  Tk_OptionTable optionTable;

  if (objc < 2) {
    Tcl_WrongNumArgs(interp, 1, objv, "pathName ?-option value ...?");
    return TCL_ERROR;
  }
    
  tkwin = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp), Tcl_GetString(objv[1]), (char *)NULL);
  if (tkwin == NULL) {
    return TCL_ERROR;
  }

  Tk_SetClass(tkwin, "Biotext");

  /*
   * Create the option table for this widget class.
   * If it has already been
   * created, the refcount will get bumped and j
   * ust the pointer will be
   * returned. The refcount getting bumped does
   * not concern us, because Tk
   * will ensure the table is deleted when the 
   * interpreter is destroyed.
   */
  optionTable = Tk_CreateOptionTable(interp, optionSpecs);

  /*
   * Initialize the widget record. 
   * The memset allows us to set
   * just the non-NULL/0 items.
   */
  BiotextPtr = (Biotext *) ckalloc(sizeof(Biotext));
  memset((void *) BiotextPtr, 0, (sizeof(Biotext)));

  /* 
   * Initialise generic options
   */
  if (Tk_InitOptions(interp, (char *) BiotextPtr, optionTable, tkwin) != TCL_OK) {
    Tk_DestroyWindow(tkwin);
    ckfree((char *) BiotextPtr);
    return TCL_ERROR;
  }
  
  /* 
   * Initialize element of the Biotext structure
   */
  BiotextPtr->tkwin       = tkwin;
  BiotextPtr->display     = Tk_Display(tkwin);
  BiotextPtr->screen      = Tk_Screen(tkwin);
  BiotextPtr->interp      = interp;
  BiotextPtr->gc          = None;
  BiotextPtr->optionTable = optionTable;
  BiotextPtr->state       = STATE_DISABLED;
  BiotextPtr->leftIndex   = 0;
  BiotextPtr->topIndex    = 0;
  BiotextPtr->inset       = 1;
  BiotextPtr->SeqMat      = NULL;
  BiotextPtr->TagMap      = NULL;
  BiotextPtr->Lock        = 1;
  BiotextPtr->widgetCmd   = Tcl_CreateObjCommand(interp, Tk_PathName(tkwin), BiotextWidgetObjCmd, BiotextPtr, (Tcl_CmdDeleteProc *)BiotextDeletedProc);

  /*
   * Set the font to be used, get font characteristics
   */
#ifdef WIN32
    BiotextPtr->xft = 0;
#else
    if (Tcl_VarEval(BiotextPtr->interp, "::tk::pkgconfig get fontsystem", NULL) == TCL_ERROR) {
      BiotextPtr->xft = 0;
    } else {
      BiotextPtr->xft = 1;
    }
#endif
  Tcl_Obj *resObj = Tcl_NewObj();
  resObj = Tcl_GetObjResult(BiotextPtr->interp);
  BiotextPtr->FontFamily = strdup("Courier");
  BiotextPtr->FontSize   = 8;
  BiotextPtr->FontWeight = strdup("normal");
  BiotextPtr->charWidth  = 1;
  BiotextPtr->charHeight = 1;
  BiotextSetFont(BiotextPtr);
  
  /*
   * Group initialisation
   */
  BiotextPtr->nbrGrps = 0;
  BiotextPtr->GroupList = NULL;
  
  /*
   * Cut and Paste initialisation
   */
  BiotextPtr->nbrSeqsCache = 0;
  BiotextPtr->SeqGrpCache = NULL;
  BiotextPtr->SeqMatCache = NULL;
  BiotextPtr->SeqTCache   = NULL;

  /*
   * set up event proc
   */
  Tk_CreateEventHandler(BiotextPtr->tkwin, ExposureMask|StructureNotifyMask|FocusChangeMask,BiotextObjEventProc, BiotextPtr);
  if (Tk_SetOptions(interp, (char *) BiotextPtr, optionTable, objc - 2, objv + 2, tkwin, NULL, NULL) != TCL_OK) {
    goto error;
  }

  if (BiotextConfigure(interp, BiotextPtr) != TCL_OK) {
    goto error;
  }

  BiotextComputeGeometry(BiotextPtr);

  Tcl_SetObjResult(interp,Tcl_NewStringObj(Tk_PathName(BiotextPtr->tkwin), -1));
  return TCL_OK;

 error:
  Tk_DestroyWindow(BiotextPtr->tkwin);
  return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * BiotextWidgetObjCmd --
 *
 *	This procedure is invoked to process the Tcl command that corresponds
 *	to a widget managed by this module. See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
BiotextWidgetObjCmd(ClientData clientData,
		    Tcl_Interp *interp,
		    int objc,
		    Tcl_Obj * const objv[]) 
{
  register Biotext *BiotextPtr = (Biotext *)clientData;
  int index, result = TCL_OK;
  Tcl_Obj *resultObjPtr;

  static CONST84 char *BiotextOptions[] = {"addseqs", "cget", "chars", "configure", "copy", "cursor", "cut", "delchars", "delete", "font", "group",  "index", "insert", "lock", "map", "mapping", "output", "paste", "push", "see", "sequences", "ungroup", "xview", "yview", NULL};
  enum selOptions {
    Biotext_ADDSEQS, Biotext_CGET, Biotext_CHARS, Biotext_CONFIGURE, Biotext_COPY, Biotext_CURSOR, Biotext_CUT, Biotext_DELCHARS, Biotext_DELETE, Biotext_FONT, Biotext_GROUP, Biotext_INDEX, Biotext_INSERT, Biotext_LOCK, Biotext_MAP, Biotext_MAPPING, Biotext_OUTPUT, Biotext_PASTE, Biotext_PUSH, Biotext_SEE, Biotext_SEQUENCES, Biotext_UNGROUP, Biotext_XVIEW, Biotext_YVIEW,
  };

  if (objc < 2) {
    Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg...?");
    return TCL_ERROR;
  }

  if (Tcl_GetIndexFromObj(interp, objv[1], BiotextOptions, "command", 0, &index) != TCL_OK) {
    return TCL_ERROR;
  }

  Tcl_Preserve((ClientData) BiotextPtr);

  switch ((enum selOptions) index) {
  case Biotext_ADDSEQS:
    if (BiotextAddSeqsCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_CGET:
    if (objc != 3) {
      Tcl_WrongNumArgs(interp, 2, objv, "option");
      goto error;
    }
    resultObjPtr = Tk_GetOptionValue(interp, (char *) BiotextPtr, BiotextPtr->optionTable, objv[2], BiotextPtr->tkwin);
    if (resultObjPtr == NULL) {
      result = TCL_ERROR;
    } else {
      Tcl_SetObjResult(interp, resultObjPtr);
    }
    break;
  case Biotext_CONFIGURE:
    resultObjPtr = NULL;
    if (objc == 2) {
      resultObjPtr = Tk_GetOptionInfo(interp, (char *) BiotextPtr,BiotextPtr->optionTable, NULL, BiotextPtr->tkwin);
      if (resultObjPtr == NULL) {
	result = TCL_ERROR;
      }
    } else if (objc == 3) {
      resultObjPtr = Tk_GetOptionInfo(interp, (char *) BiotextPtr, BiotextPtr->optionTable, objv[2], BiotextPtr->tkwin);
      if (resultObjPtr == NULL) {
	result = TCL_ERROR;
      }
    } else {
      result = Tk_SetOptions(interp, (char *) BiotextPtr, BiotextPtr->optionTable, objc - 2, objv + 2,BiotextPtr->tkwin, NULL, NULL);
      if (result == TCL_OK) {
	result = BiotextConfigure(interp, BiotextPtr);
      }
      BiotextEventuallyRedraw(BiotextPtr);
    }
    if (resultObjPtr != NULL) {
      Tcl_SetObjResult(interp, resultObjPtr);
    }
    break;
    /* 
       case Biotext_SELECTION:
       if (objc < 3) {
       Tcl_WrongNumArgs(interp, 3, objv, "option arg ?arg?");
       goto error;
       }
       BiotextSelectionCmd(BiotextPtr, interp, objc, objv);
       break;
    */
  case Biotext_SEQUENCES:
    BiotextSequencesCmd(BiotextPtr, interp, objc, objv);
    break;
  case Biotext_INSERT:
    if (BiotextInsertCmd(BiotextPtr, interp, objc, objv) == TCL_ERROR) goto error;
    break;
  case Biotext_DELETE:
    if (BiotextDeleteCmd(BiotextPtr, interp, objc, objv) == TCL_ERROR) goto error;
    break;
  case Biotext_MAPPING:
    BiotextMappingCmd(BiotextPtr, interp, objc,objv);
    break;
  case Biotext_MAP:
    BiotextMapCmd(BiotextPtr, interp, objc,objv);
    break;
  case Biotext_XVIEW:
    if (BiotextXViewCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_YVIEW:
    if (BiotextYViewCmd(BiotextPtr, interp, objc, objv)) goto error;
    break;
  case Biotext_GROUP:
    if (BiotextGroupCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_UNGROUP:
    if (BiotextUngroupCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_SEE:
    if (BiotextSeeCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_INDEX:
    if (BiotextIndexCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_PUSH:
    if (BiotextPushCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_CURSOR:
    if (BiotextCursorCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_CUT:
    if (BiotextCutCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_COPY:
    if (BiotextCopyCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_PASTE:
    if (BiotextPasteCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_FONT:
    if (BiotextFontCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_OUTPUT:
    if (BiotextOutputCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_DELCHARS:
    if (BiotextDelcharsCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_LOCK:
    if (BiotextLockCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
    break;
  case Biotext_CHARS:
    if (BiotextCharsCmd(BiotextPtr, interp, objc, objv) != TCL_OK) goto error;
  }
    
  Tcl_Release(BiotextPtr);
  return result;

 error:
  Tcl_Release(BiotextPtr);
  return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextConfigCursor --
 *	Configures the timer depending on the 
 *      state of the Biotext widget.
 *	Equivalent to BiotextFocusProc.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The cursor will be switched off/on.
 *
 *----------------------------------------------------------------------
 */
void
BiotextConfigCursor(register Biotext *BiotextPtr)
{
  if (BiotextPtr->state == STATE_DISABLED) 
    return;
  
  /*
   * To have a cursor, we have to have focus and 
   * allow edits
   */
  if (BiotextPtr->flags & GOT_FOCUS) {
    /*
     * Turn the cursor ON
     */
    if (! (BiotextPtr->flags & CURSOR_ON)) {
      BiotextPtr->flags |= CURSOR_ON;
    }
    
    /* set up the first timer */
    if (BiotextPtr->insertOffTime != 0) {
      /* make sure nothing existed */
      Tcl_DeleteTimerHandler(BiotextPtr->cursorTimer);
      BiotextPtr->cursorTimer =	Tcl_CreateTimerHandler(BiotextPtr->insertOnTime, BiotextCursorEvent, (ClientData) BiotextPtr);
    }
  } else {
    /*
     * Turn the cursor OFF
     */
    if ((BiotextPtr->flags & CURSOR_ON)) {
      BiotextPtr->flags &= ~CURSOR_ON;
    }
    
    /* and disable the timer */
    if (BiotextPtr->cursorTimer != NULL) {
      Tcl_DeleteTimerHandler(BiotextPtr->cursorTimer);
    }
    BiotextPtr->cursorTimer = NULL;
  }  

  //BiotextEventuallyRedraw(BiotextPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextConfigure --
 *
 *	This procedure is called to process an 
 *      argv/argc list in conjunction
 *	with the Tk option database to configure (or 
 *      reconfigure) a Biotext
 *	widget.
 *
 * Results:
 *	The return value is a standard Tcl result. 
 *      If TCL_ERROR is returned, then the 
 *      interp's result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as 
 *      colors, border width, etc. get set
 *	for BiotextPtr; old resources get freed, 
 *      if there were any.
 *
 *----------------------------------------------------------------------
 */

int
BiotextConfigure(Tcl_Interp *interp, 
		 Biotext *BiotextPtr)
{
  Tk_3DBorder bgBorder;
  int doubleBuffer;
  
  /*
   * Set the background for the window and create a 
   * graphics context for use during redisplay.
   */
  bgBorder = BiotextPtr->bgBorder;
  Tk_SetWindowBackground(BiotextPtr->tkwin,Tk_3DBorderColor(bgBorder)->pixel);
  
  Tcl_GetBooleanFromObj(NULL, BiotextPtr->doubleBufferPtr, &doubleBuffer);
  if ((BiotextPtr->gc == None) && (doubleBuffer)) {
    XGCValues gcValues;
    gcValues.function = GXcopy;
    gcValues.graphics_exposures = False;
    gcValues.font = Tk_FontId(BiotextPtr->tkfont);
    BiotextPtr->gc = Tk_GetGC(BiotextPtr->tkwin,GCFunction|GCGraphicsExposures|GCFont, &gcValues);
  }
    
  /*
   * Setup the blinking cursor.
   */
  BiotextComputeGeometry(BiotextPtr);
  BiotextConfigCursor(BiotextPtr);

  BiotextEventuallyRedraw(BiotextPtr);

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * BiotextObjEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various events on
 *	Biotexts.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get cleaned up. When
 *	it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

void
BiotextObjEventProc(
		    ClientData clientData,	/* Information about window. */
		    XEvent *eventPtr)		/* Information about event. */
{
  register Biotext *BiotextPtr = clientData;

  switch (eventPtr->type) {
  case Expose:
    BiotextEventuallyRedraw(BiotextPtr);
    break;
  case ConfigureNotify:
    /*
     * Adjust the geometry.
     * Then arrange for the window to be redisplayed.
     */    
    Tcl_Preserve((ClientData) BiotextPtr);
    BiotextComputeGeometry(BiotextPtr);
    Tk_SetInternalBorder(BiotextPtr->tkwin, BiotextPtr->borderWidth);

    //BiotextPtr->flags |= UPDATE_SCROLLBAR;
    //BiotextEventuallyRedraw(BiotextPtr);

    Tcl_Release((ClientData) BiotextPtr);
    break;
  case DestroyNotify:
    if (BiotextPtr->tkwin != NULL) {
      Tk_FreeConfigOptions((char *) BiotextPtr, BiotextPtr->optionTable,BiotextPtr->tkwin);
      if (BiotextPtr->gc != None) {
	Tk_FreeGC(BiotextPtr->display, BiotextPtr->gc);
      }
      BiotextPtr->tkwin = NULL;
      Tcl_DeleteCommandFromToken(BiotextPtr->interp,BiotextPtr->widgetCmd);
      break;
    case FocusIn:
    case FocusOut:
      if (eventPtr->xfocus.detail != NotifyInferior) {
	if (eventPtr->type == FocusOut) {
	  BiotextPtr->flags &= ~GOT_FOCUS;
	} else {
	  BiotextPtr->flags |= GOT_FOCUS;
	}
	/* cancel the timer */
	BiotextConfigCursor(BiotextPtr);
      }
      break;
    }
  }
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted. If the
 *	widget isn't already in the process of being destroyed, this command
 *	destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */

void
BiotextDeletedProc(
		   ClientData clientData)	/* Pointer to widget record for widget. */
{
  Biotext *BiotextPtr = clientData;
  Tk_Window tkwin = BiotextPtr->tkwin;

  /*
   * This procedure could be invoked either because the window was destroyed
   * and the command was then deleted (in which case tkwin is NULL) or
   * because the command was deleted, and then this procedure destroys the
   * widget.
   */

  if (tkwin != NULL) {
    Tk_DestroyWindow(tkwin);
  }
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextEventuallyRedraw --
 *
 *	Ensure that a biotext is eventually redrawn 
 *      on the display.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets redisplayed. Right now we 
 *      don't do selective redisplays: the whole 
 *      window will be redrawn.
 *      This doesn't seem to hurt performance 
 *      noticeably, but if it does then this could be
 *	changed.
 *
 *----------------------------------------------------------------------
 */
void
BiotextEventuallyRedraw(Biotext *BiotextPtr) 
{
  if ((BiotextPtr->flags & BIOTEXT_DELETED) || ! Tk_IsMapped(BiotextPtr->tkwin)) {
    return;
  }

  /*
   * Right now we don't do selective redisplays: 
   * the whole window will be redrawn. This doesn't 
   * seem to hurt performance noticeably, but if it
   * does then this could be changed.
   */

  if (!(BiotextPtr->flags & REDRAW_PENDING)) {
    BiotextPtr->flags |= REDRAW_PENDING;

    Tcl_DoWhenIdle(BiotextDisplay, (ClientData) BiotextPtr);
  }
}

/*
 *--------------------------------------------------------------
 *
 * BiotextDisplay --
 *
 *	This procedure redraws the contents of a Biotext window. It is invoked
 *	as a do-when-idle handler, so it only runs when there's nothing else
 *	for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

void
BiotextDisplay(ClientData clientData) 
{
  register Biotext *BiotextPtr = (Biotext *) clientData;
  Tk_Window tkwin = BiotextPtr->tkwin;
  Display *display = BiotextPtr->display;
  Pixmap pm = None;
  Drawable d;
  int relief, doubleBuffer, width, height;
  Tk_3DBorder bgBorder;

  /*
   * reset the pending flag
   */
  BiotextPtr->flags &= ~REDRAW_PENDING;
  if (tkwin == NULL || ! Tk_IsMapped(tkwin)) {
    return;
  }

  /*
   * Update the scrollbar if that's needed.
   */
  if (BiotextPtr->flags & UPDATE_SCROLLBAR) {
    BiotextPtr->flags &= ~UPDATE_SCROLLBAR;

    /*
     * Preserve/Release because updating the 
     * scrollbar can have the side-effect of 
     * destroying or unmapping the biotext widget.
     */
    Tcl_Preserve(BiotextPtr);

    BiotextUpdateXScrollbar(BiotextPtr);
    BiotextUpdateYScrollbar(BiotextPtr);
    if ((BiotextPtr->flags & BIOTEXT_DELETED) || !Tk_IsMapped(tkwin)) {
      Tcl_Release(BiotextPtr);
      return;
    }
    Tcl_Release(BiotextPtr);
  }

  /*
   * Compute the geometry and adjust it so it matches
   * an integer number of characters.
   */
  //BiotextComputeGeometry(BiotextPtr);  
  width  = BiotextPtr->width;
  height = BiotextPtr->height;
  Tk_GeometryRequest(tkwin, width, height);

  /*
   * Create a pixmap for doublebuffering if necessary.
   */
  Tcl_GetBooleanFromObj(NULL, BiotextPtr->doubleBufferPtr, &doubleBuffer);
  //doubleBuffer = 0;
  if (doubleBuffer) {
    pm = Tk_GetPixmap(display, Tk_WindowId(tkwin), width, height, DefaultDepthOfScreen(Tk_Screen(tkwin)));
    d = pm;
  } else {
    d = Tk_WindowId(tkwin);
  }
  Tmp_Color(BiotextPtr, d);

  /*
   * Redraw the widget's background and border.
   */
  bgBorder = BiotextPtr->bgBorder;
  Tk_GetReliefFromObj(NULL, BiotextPtr->reliefPtr, &relief);

  Tk_Fill3DRectangle(tkwin, d, bgBorder, 0, 0, width, height, BiotextPtr->borderWidth, relief);

  /*
   * Display the Biotext.
   *
   * Set up the graphical context:
   * - defaultgc is the one used when no mapping 
   */
  GC newgc, defaultgc;
  XGCValues gcValues;

  gcValues.function = GXcopy;
  gcValues.graphics_exposures = False;
  gcValues.font = Tk_FontId(BiotextPtr->tkfont);
  newgc = XCreateGC(display, d, GCFunction|GCGraphicsExposures|GCFont, &gcValues);

  gcValues.foreground = Tk_GetColor(BiotextPtr->interp, tkwin, "black")->pixel;
  gcValues.background = Tk_GetColor(BiotextPtr->interp, tkwin, "white")->pixel;
  defaultgc = XCreateGC(display, d, GCFunction|GCGraphicsExposures|GCFont|GCForeground|GCBackground, &gcValues);

  /*
   * If Empty widget, then return.
   */
  if (BiotextPtr->SeqMat == NULL) {
    if (doubleBuffer) {
      XCopyArea(display, pm, Tk_WindowId(tkwin), newgc, 0, 0, (unsigned) width, (unsigned) height, 0, 0);
      Tk_FreePixmap(display, pm);
    }
    
    return;
  }

  /* Draw the character matrix according to the 
   * current mapping.
   */
  int xP , yP, wc, hc;
  BiotextDrawAlignment(BiotextPtr, d);

  /*
   * Draw the blinking block cursor if any
   */
  if ((BiotextPtr->state == STATE_NORMAL) && (BiotextPtr->flags & GOT_FOCUS) && (BiotextPtr->flags & CURSOR_ON)) {
    int col, row;

    wc = BiotextPtr->charWidth;
    hc = BiotextPtr->charHeight;
    col = BiotextPtr->currentC;
    row = BiotextPtr->currentR;
    if (Biotext_IsVisible(BiotextPtr, col, row, 1)) {
      Biotext_Coords2Pixels(BiotextPtr, col, row, &xP, &yP, 1);

      Tk_Fill3DRectangle(tkwin, d, Tk_Get3DBorder(BiotextPtr->interp, tkwin, Tk_NameOfColor(BiotextPtr->cursorColor)), xP, yP, wc, hc, 1, TK_RELIEF_RAISED);
    }
  }

  /*
   * If double-buffered, copy to the screen and 
   * release the pixmap.
   */
  if (doubleBuffer) {
    gcValues.foreground = Tk_GetColor(BiotextPtr->interp, tkwin, "black")->pixel;
    gcValues.background = Tk_GetColor(BiotextPtr->interp, tkwin, "darkslategrey")->pixel;
    XChangeGC(display, newgc, GCForeground|GCBackground, &gcValues);
    XCopyArea(display, pm, Tk_WindowId(tkwin), newgc, 0, 0, (unsigned) width, (unsigned) height, 0, 0);
    Tk_FreePixmap(display, pm);
  }
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextDestroy --
 *
 *	This procedure is invoked by 
 *      Tcl_EventuallyFree or Tcl_Release to
 *	clean up the internal structure of a Biotext 
 *      at a safe time when no-one is using it anymore
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the Biotext is 
 *      freed up.
 *
 *----------------------------------------------------------------------
 */

void
BiotextDestroy(char *memPtr) /* Info about Biotext widget. */
{
  Biotext *BiotextPtr = (Biotext *) memPtr;

  ckfree((char *) BiotextPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextVisibleXRange --
 *
 *	Return information about the range of the 
 *      Biotext in X-coord that is currently
 *	visible.
 *
 * Results:
 *	*firstPtr and *lastPtr are modified to hold 
 *      fractions between 0 and 1
 *	identifying the range of characters visible 
 *      in the Biotext.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
BiotextVisibleXRange(
		     Biotext *BiotextPtr,
		     double *firstPtr,  
		     double *lastPtr) 
{
  int dif;
  
  if (BiotextPtr->lgSeqs == 0) {
    *firstPtr = 0.0;
    *lastPtr = 1.0;
  } else {
    dif = BiotextPtr->lgSeqs - (BiotextPtr->leftIndex + BiotextPtr->widthChar);
    if (BiotextPtr->lgSeqs <= BiotextPtr->widthChar) {
      BiotextPtr->leftIndex = 0;
    } else if (dif < 0) {
      BiotextPtr->leftIndex += dif;
    }

    *firstPtr = (double) BiotextPtr->leftIndex / BiotextPtr->lgSeqs;
    *lastPtr = (double) (BiotextPtr->leftIndex + BiotextPtr->widthChar) / BiotextPtr->lgSeqs;
  }
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextVisibleYRange --
 *
 *	Return information about the range of the 
 *      Biotext in Y-coord that is currently
 *	visible.
 *
 * Results:
 *	*firstPtr and *lastPtr are modified to hold 
 *      fractions between 0 and 1
 *	identifying the range of characters visible 
 *      in the Biotext.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
BiotextVisibleYRange(
		     Biotext *BiotextPtr,
		     double *firstPtr,
		     double *lastPtr) 
{
  int dif;
  
  if (BiotextPtr->nbSeqs == 0) {
    *firstPtr = 0.0;
    *lastPtr = 1.0;
  } else {
    dif = BiotextPtr->nbSeqs - (BiotextPtr->topIndex + BiotextPtr->heightChar);
    if (BiotextPtr->nbSeqs<= BiotextPtr->heightChar) {
      BiotextPtr->topIndex = 0;
    } else if (dif < 0) {
      BiotextPtr->topIndex += dif;
    }

    *firstPtr = (double) BiotextPtr->topIndex / BiotextPtr->nbSeqs;
    *lastPtr = (double) (BiotextPtr->topIndex + BiotextPtr->heightChar) / (double) BiotextPtr->nbSeqs;
  }
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextUpdateXScrollbar --
 *
 *	This function is invoked whenever 
 *      information has changed in an Biotext
 *	in a way that would invalidate a X -xscrollbar
 *      display. If there is an
 *	associated X -scrollbar, then this function 
 *      updates it by invoking a Tcl
 *	command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A Tcl command is invoked, and an additional 
 *      command may be
 *	invoked to process errors in the command.
 *
 *----------------------------------------------------------------------
 */

void
BiotextUpdateXScrollbar(Biotext *BiotextPtr) 
{
  char firstStr[TCL_DOUBLE_SPACE], lastStr[TCL_DOUBLE_SPACE];
  int code;
  double first, last;
  Tcl_Interp *interp;

  if (BiotextPtr->xscrollCmd == NULL) {
    return;
  }

  interp = BiotextPtr->interp;
  Tcl_Preserve(interp);

  BiotextVisibleXRange(BiotextPtr, &first, &last);
  Tcl_PrintDouble(NULL, first, firstStr);
  Tcl_PrintDouble(NULL, last, lastStr);
  code = Tcl_VarEval(interp, BiotextPtr->xscrollCmd, " ", firstStr, " ", lastStr, NULL);
  if (code != TCL_OK) {
    Tcl_AddErrorInfo(interp, "\n    (horizontal scrolling command executed by ");
    Tcl_AddErrorInfo(interp, Tk_PathName(BiotextPtr->tkwin));
    Tcl_AddErrorInfo(interp, ")");
    Tcl_BackgroundError(interp);
  }

  Tcl_ResetResult(interp);
  Tcl_Release(interp);
}


/*
 *----------------------------------------------------------------------
 *
 * BiotextUpdateYScrollbar --
 *
 *	This function is invoked whenever 
 *      information has changed in an Biotext
 *	in a way that would invalidate an
 *      Y -yxscrollbar display. If there is an
 *	associated X -scrollbar, then this function 
 *      updates it by invoking a Tcl
 *	command.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A Tcl command is invoked, and an additional command may be
 *	invoked to process errors in the command.
 *
 *----------------------------------------------------------------------
 */

void
BiotextUpdateYScrollbar(Biotext *BiotextPtr) 
{
  char firstStr[TCL_DOUBLE_SPACE], lastStr[TCL_DOUBLE_SPACE];
  int code;
  double first, last;
  Tcl_Interp *interp;
  
  if (BiotextPtr->yscrollCmd == NULL) {
    return;
  }
  
  interp = BiotextPtr->interp;
  Tcl_Preserve(interp);
  
  BiotextVisibleYRange(BiotextPtr, &first, &last);
  Tcl_PrintDouble(NULL, first, firstStr);
  Tcl_PrintDouble(NULL, last, lastStr);
  code = Tcl_VarEval(interp, BiotextPtr->yscrollCmd, " ", firstStr, " ", lastStr, NULL);
  if (code != TCL_OK) {
    Tcl_AddErrorInfo(interp, "\n    (horizontal scrolling command executed by ");
    Tcl_AddErrorInfo(interp, Tk_PathName(BiotextPtr->tkwin));
    Tcl_AddErrorInfo(interp, ")");
    Tcl_BackgroundError(interp);
  }
  
  Tcl_ResetResult(interp);
  Tcl_Release(interp);
}

/*
 *----------------------------------------------------------------------
 *
 * BiotextCursorEvent --
 *	Toggle the cursor status. 
 *      Equivalent to EntryBlinkProc.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The cursor will be switched off/on.
 *
 *----------------------------------------------------------------------
 */
void
BiotextCursorEvent(ClientData clientData)
{
  register Biotext *BiotextPtr = (Biotext *) clientData;

  if (! (BiotextPtr->flags & GOT_FOCUS) || (BiotextPtr->insertOffTime == 0) || (BiotextPtr->state != STATE_NORMAL)) {
    return;
  }
  
  if (BiotextPtr->cursorTimer != NULL) {
    Tcl_DeleteTimerHandler(BiotextPtr->cursorTimer);
  }

  BiotextPtr->cursorTimer = Tcl_CreateTimerHandler((BiotextPtr->flags & CURSOR_ON) ? BiotextPtr->insertOffTime : BiotextPtr->insertOnTime, BiotextCursorEvent, (ClientData) BiotextPtr);

  /* Toggle the cursor */
  BiotextPtr->flags ^= CURSOR_ON;

  /* Redraw */
  BiotextEventuallyRedraw(BiotextPtr);
}

/*
 * Package initialisation
 */
int DLLEXPORT Biotext_Init(Tcl_Interp *interp) 
{
  if (Tcl_InitStubs(interp, "8.0" , 0) == NULL) {
    return TCL_ERROR;
  }
  if (Tk_InitStubs(interp, "8.0" , 0) == NULL) {
    return TCL_ERROR;
  }
  
  Tcl_CreateObjCommand(interp, "biotext", BiotextObjCmd, (ClientData) NULL, (Tcl_CmdDeleteProc *)NULL);

  Tcl_PkgProvide(interp,"biotext","0.1");

  return TCL_OK;
}









