/*
 * tkSquare.c --
 *
 *	This module implements "square" widgets that are object based. A
 *	"square" is a widget that displays a single square that can be moved
 *	around and resized. This file is intended as an example of how to
 *	build a widget; it isn't included in the normal wish, but it is
 *	included in "tktest".
 *
 * Copyright (c) 1997 Sun Microsystems, Inc.
*
* See the file "license.terms" for information on usage and redistribution of
* this file, and for a DISCLAIMER OF ALL WARRANTIES.
*
* RCS: @(#) $Id: tkSquare.c,v 1.13 2008/10/17 23:18:37 nijtmans Exp $
*/

#include <tk.h>
#include "tkBiotext.h"
#include <tkFont.h>
#include <tkWinInt.h>

#ifndef DLLEXPORT
#define DLLEXPORT __declspec(dllexport)
#endif 

/* Taken from tk8.6.0b1/win/tkWinFont.c */
/*
 * The following structure represents a font family. It is assumed that all
 * screen fonts constructed from the same "font family" share certain
 * properties; all screen fonts with the same "font family" point to a shared
 * instance of this structure. The most important shared property is the
 * character existence metrics, used to determine if a screen font can display
 * a given Unicode character.
 *
 * Under Windows, a "font family" is uniquely identified by its face name.
 */

#define FONTMAP_SHIFT	    10

#define FONTMAP_PAGES	    	(1 << (sizeof(Tcl_UniChar)*8 - FONTMAP_SHIFT))
#define FONTMAP_BITSPERPAGE	(1 << FONTMAP_SHIFT)

typedef struct FontFamily {
    struct FontFamily *nextPtr;	/* Next in list of all known font families. */
    int refCount;		/* How many SubFonts are referring to this
				 * FontFamily. When the refCount drops to
				 * zero, this FontFamily may be freed. */
    /*
     * Key.
     */

    Tk_Uid faceName;		/* Face name key for this FontFamily. */

    /*
     * Derived properties.
     */

    Tcl_Encoding encoding;	/* Encoding for this font family. */
    int isSymbolFont;		/* Non-zero if this is a symbol font. */
    int isWideFont;		/* 1 if this is a double-byte font, 0
				 * otherwise. */
    BOOL (WINAPI *textOutProc)(HDC hdc, int x, int y, TCHAR *str, int len);
				/* The procedure to use to draw text after it
				 * has been converted from UTF-8 to the
				 * encoding of this font. */
    BOOL (WINAPI *getTextExtentPoint32Proc)(HDC, TCHAR *, int, LPSIZE);
				/* The procedure to use to measure text after
				 * it has been converted from UTF-8 to the
				 * encoding of this font. */

    char *fontMap[FONTMAP_PAGES];
				/* Two-level sparse table used to determine
				 * quickly if the specified character exists.
				 * As characters are encountered, more pages
				 * in this table are dynamically added. The
				 * contents of each page is a bitmask
				 * consisting of FONTMAP_BITSPERPAGE bits,
				 * representing whether this font can be used
				 * to display the given character at the
				 * corresponding bit position. The high bits
				 * of the character are used to pick which
				 * page of the table is used. */

    /*
     * Cached Truetype font info.
     */

    int segCount;		/* The length of the following arrays. */
    USHORT *startCount;		/* Truetype information about the font, */
    USHORT *endCount;		/* indicating which characters this font can
				 * display (malloced). The format of this
				 * information is (relatively) compact, but
				 * would take longer to search than indexing
				 * into the fontMap[][] table. */
} FontFamily;

/*
 * The following structure encapsulates an individual screen font. A font
 * object is made up of however many SubFonts are necessary to display a
 * stream of multilingual characters.
 */

typedef struct SubFont {
    char **fontMap;		/* Pointer to font map from the FontFamily,
				 * cached here to save a dereference. */
    HFONT hFont0;		/* The specific screen font that will be used
				 * when displaying/measuring chars belonging
				 * to the FontFamily. */
    FontFamily *familyPtr;	/* The FontFamily for this SubFont. */
    HFONT hFontAngled;
    double angle;
} SubFont;

/*
 * The following structure represents Windows' implementation of a font
 * object.
 */

#define SUBFONT_SPACE		3
#define BASE_CHARS		128

typedef struct WinFont {
    TkFont font;		/* Stuff used by generic font package. Must be
				 * first in structure. */
    SubFont staticSubFonts[SUBFONT_SPACE];
				/* Builtin space for a limited number of
				 * SubFonts. */
    int numSubFonts;		/* Length of following array. */
    SubFont *subFontArray;	/* Array of SubFonts that have been loaded in
				 * order to draw/measure all the characters
				 * encountered by this font so far. All fonts
				 * start off with one SubFont initialized by
				 * AllocFont() from the original set of font
				 * attributes. Usually points to
				 * staticSubFonts, but may point to malloced
				 * space if there are lots of SubFonts. */
    HWND hwnd;			/* Toplevel window of application that owns
				 * this font, used for getting HDC for
				 * offscreen measurements. */
    int pixelSize;		/* Original pixel size used when font was
				 * constructed. */
    int widths[BASE_CHARS];	/* Widths of first 128 chars in the base font,
				 * for handling common case. The base font is
				 * always used to draw characters between
				 * 0x0000 and 0x007f. */
} WinFont;

/*
 * A data structure of the following type is kept for each square widget
 * managed by this file:
 */

typedef struct {
  int charWidth;
  int charHeight;
  int ascent;
  int descent;
  Tk_Font tkfont;
  Tcl_Obj *tkfontPtr;

  Tk_Window tkwin;		/* Window that embodies the square. NULL means
				 * window has been deleted but widget record
				 * hasn't been cleaned up yet. */
  Display *display;		/* X's token for the window's display. */
  Tcl_Interp *interp;		/* Interpreter associated with widget. */
  Tcl_Command widgetCmd;	/* Token for square's widget command. */
  Tk_OptionTable optionTable;	/* Token representing the configuration
				 * specifications. */
  Tcl_Obj *xPtr, *yPtr;	/* Position of square's upper-left corner
			 * within widget. */
  int x, y;
  Tcl_Obj *sizeObjPtr;	/* Width and height of square. */

  /*
   * Information used when displaying widget:
   */

  Tcl_Obj *borderWidthPtr;	/* Width of 3-D border around whole widget. */
  Tcl_Obj *bgBorderPtr;
  Tcl_Obj *fgBorderPtr;
  Tcl_Obj *reliefPtr;
  GC gc;			/* Graphics context for copying from
				 * off-screen pixmap onto screen. */
  Tcl_Obj *doubleBufferPtr;	/* Non-zero means double-buffer redisplay with
				 * pixmap; zero means draw straight onto the
				 * display. */
  int updatePending;		/* Non-zero means a call to SquareDisplay has
				 * already been scheduled. */
} Square;

/*
 * Information used for argv parsing.
 */

static const Tk_OptionSpec optionSpecs[] = {
  {TK_OPTION_BORDER, "-background", "background", "Background", "#d9d9d9", Tk_Offset(Square, bgBorderPtr), -1, 0, (ClientData) "white"},
  {TK_OPTION_SYNONYM, "-bg", NULL, NULL, NULL, 0, -1, 0, (ClientData) "-background"},
  {TK_OPTION_FONT, "-font", "font", "Font", "Courier 20", Tk_Offset(Square, tkfontPtr), -1, 0, (ClientData) "Courier 20"},
  {TK_OPTION_SYNONYM, "-bd", NULL, NULL, NULL, 0, -1, 0, (ClientData) "-borderwidth"},
  {TK_OPTION_PIXELS, "-borderwidth", "borderWidth", "BorderWidth", "2", Tk_Offset(Square, borderWidthPtr), -1},
  {TK_OPTION_BOOLEAN, "-dbl", "doubleBuffer", "DoubleBuffer", "1", Tk_Offset(Square, doubleBufferPtr), -1},
  {TK_OPTION_SYNONYM, "-fg", NULL, NULL, NULL, 0, -1, 0, (ClientData) "-foreground"},
  {TK_OPTION_BORDER, "-foreground", "foreground", "Foreground", "#b03060", Tk_Offset(Square, fgBorderPtr), -1, 0, (ClientData) "black"},
  {TK_OPTION_PIXELS, "-posx", "posx", "PosX", "0",
   Tk_Offset(Square, xPtr), -1},
  {TK_OPTION_PIXELS, "-posy", "posy", "PosY", "0",
   Tk_Offset(Square, yPtr), -1},
  {TK_OPTION_RELIEF, "-relief", "relief", "Relief",
   "raised", Tk_Offset(Square, reliefPtr), -1},
  {TK_OPTION_PIXELS, "-size", "size", "Size", "20",
   Tk_Offset(Square, sizeObjPtr), -1},
  {TK_OPTION_END}
};

/*
 * Forward declarations for procedures defined later in this file:
 */

int			SquareObjCmd(ClientData clientData,
				     Tcl_Interp *interp, int objc,
				     Tcl_Obj * const objv[]);
static void		SquareDeletedProc(ClientData clientData);
static int		SquareConfigure(Tcl_Interp *interp, Square *squarePtr);
static void		SquareDestroy(char *memPtr);
static void		SquareDisplay(ClientData clientData);
static void		KeepInWindow(Square *squarePtr);
static void		SquareObjEventProc(ClientData clientData,
					   XEvent *eventPtr);
static int		SquareWidgetObjCmd(ClientData clientData,
					   Tcl_Interp *, int objc, Tcl_Obj * const objv[]);

void
squareDrawImageString(Square *squarePtr,
		      Display *display,
		      Drawable d,
		      GC gc,
		      int x, int y, 
		      char *string,
		      int length);

void
MyFontInfoSq(Square *squarePtr);




/*
 *--------------------------------------------------------------
 *
 * SquareObjCmd --
 *
 *	This procedure is invoked to process the "square" Tcl command. It
 *	creates a new "square" widget.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	A new widget is created and configured.
 *
 *--------------------------------------------------------------
 */

int
SquareObjCmd(
	     ClientData clientData,	/* NULL. */
	     Tcl_Interp *interp,		/* Current interpreter. */
	     int objc,			/* Number of arguments. */
	     Tcl_Obj *const objv[])	/* Argument objects. */
{
  Square *squarePtr;
  Tk_Window tkwin;
  Tk_OptionTable optionTable;
    
  if (objc < 2) {
    Tcl_WrongNumArgs(interp, 1, objv, "pathName ?-option value ...?");
    return TCL_ERROR;
  }

  tkwin = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp), Tcl_GetString(objv[1]), NULL);
  if (tkwin == NULL) {
    return TCL_ERROR;
  }
  Tk_SetClass(tkwin, "Square");

  /*
   * Create the option table for this widget class. If it has already been
   * created, the refcount will get bumped and just the pointer will be
   * returned. The refcount getting bumped does not concern us, because Tk
   * will ensure the table is deleted when the interpreter is destroyed.
   */

  optionTable = Tk_CreateOptionTable(interp, optionSpecs);

  /*
   * Allocate and initialize the widget record. The memset allows us to set
   * just the non-NULL/0 items.
   */

  squarePtr = (Square *) ckalloc(sizeof(Square));
  memset((void *) squarePtr, 0, (sizeof(Square)));

  squarePtr->tkwin     = tkwin;
  squarePtr->display   = Tk_Display(tkwin);
  squarePtr->interp    = interp;
  squarePtr->widgetCmd = Tcl_CreateObjCommand(interp,	Tk_PathName(tkwin), SquareWidgetObjCmd, squarePtr, SquareDeletedProc);
  squarePtr->gc = None;
  squarePtr->optionTable = optionTable;

  if (Tk_InitOptions(interp, (char *) squarePtr, optionTable, tkwin)
      != TCL_OK) {
    Tk_DestroyWindow(squarePtr->tkwin);
    ckfree((char *) squarePtr);
    return TCL_ERROR;
  }

  Tk_CreateEventHandler(squarePtr->tkwin, ExposureMask|StructureNotifyMask,
			SquareObjEventProc, squarePtr);
  if (Tk_SetOptions(interp, (char *) squarePtr, optionTable, objc - 2,
		    objv + 2, tkwin, NULL, NULL) != TCL_OK) {
    goto error;
  }
  if (SquareConfigure(interp, squarePtr) != TCL_OK) {
    goto error;
  }

  Tcl_SetObjResult(interp, Tcl_NewStringObj(Tk_PathName(squarePtr->tkwin), -1));
  return TCL_OK;

 error:
  Tk_DestroyWindow(squarePtr->tkwin);
  return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * SquareWidgetObjCmd --
 *
 *	This procedure is invoked to process the Tcl command that corresponds
 *	to a widget managed by this module. See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
SquareWidgetObjCmd(
		   ClientData clientData,	/* Information about square widget. */
		   Tcl_Interp *interp,		/* Current interpreter. */
		   int objc,			/* Number of arguments. */
		   Tcl_Obj * const objv[])	/* Argument objects. */
{
  Square *squarePtr = clientData;
  int result = TCL_OK;
  static const char *const squareOptions[] = {"cget", "configure", NULL};
  enum {
    SQUARE_CGET, SQUARE_CONFIGURE
  };
  Tcl_Obj *resultObjPtr;
  int index;

  if (objc < 2) {
    Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg...?");
    return TCL_ERROR;
  }

  if (Tcl_GetIndexFromObj(interp, objv[1], squareOptions, "command",
			  0, &index) != TCL_OK) {
    return TCL_ERROR;
  }

  Tcl_Preserve(squarePtr);

  switch (index) {
  case SQUARE_CGET:
    if (objc != 3) {
      Tcl_WrongNumArgs(interp, 2, objv, "option");
      goto error;
    }
    resultObjPtr = Tk_GetOptionValue(interp, (char *) squarePtr,
				     squarePtr->optionTable, objv[2], squarePtr->tkwin);
    if (resultObjPtr == NULL) {
      result = TCL_ERROR;
    } else {
      Tcl_SetObjResult(interp, resultObjPtr);
    }
    break;
  case SQUARE_CONFIGURE:
    resultObjPtr = NULL;
    if (objc == 2) {
      resultObjPtr = Tk_GetOptionInfo(interp, (char *) squarePtr,
				      squarePtr->optionTable, NULL, squarePtr->tkwin);
      if (resultObjPtr == NULL) {
	result = TCL_ERROR;
      }
    } else if (objc == 3) {
      resultObjPtr = Tk_GetOptionInfo(interp, (char *) squarePtr,
				      squarePtr->optionTable, objv[2], squarePtr->tkwin);
      if (resultObjPtr == NULL) {
	result = TCL_ERROR;
      }
    } else {
      result = Tk_SetOptions(interp, (char *) squarePtr,
			     squarePtr->optionTable, objc - 2, objv + 2,
			     squarePtr->tkwin, NULL, NULL);
      if (result == TCL_OK) {
	result = SquareConfigure(interp, squarePtr);
      }
      if (!squarePtr->updatePending) {
	Tcl_DoWhenIdle(SquareDisplay, squarePtr);
	squarePtr->updatePending = 1;
      }
    }
    if (resultObjPtr != NULL) {
      Tcl_SetObjResult(interp, resultObjPtr);
    }
  }
  Tcl_Release(squarePtr);
  return result;

 error:
  Tcl_Release(squarePtr);
  return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * SquareConfigure --
 *
 *	This procedure is called to process an argv/argc list in conjunction
 *	with the Tk option database to configure (or reconfigure) a square
 *	widget.
 *
 * Results:
 *	The return value is a standard Tcl result. If TCL_ERROR is returned,
 *	then the interp's result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width, etc. get set
 *	for squarePtr; old resources get freed, if there were any.
 *
 *----------------------------------------------------------------------
 */

static int
SquareConfigure(
		Tcl_Interp *interp,		/* Used for error reporting. */
		Square *squarePtr)		/* Information about widget. */
{
  int borderWidth;
  Tk_3DBorder bgBorder;
  int doubleBuffer;

  /*
   * Set the background for the window and create a graphics context for use
   * during redisplay.
   */

  squarePtr->tkfont = Tk_AllocFontFromObj(interp, squarePtr->tkwin, squarePtr->tkfontPtr);
  bgBorder = Tk_Get3DBorderFromObj(squarePtr->tkwin,
				   squarePtr->bgBorderPtr);
  Tk_SetWindowBackground(squarePtr->tkwin,
			 Tk_3DBorderColor(bgBorder)->pixel);
  Tcl_GetBooleanFromObj(NULL, squarePtr->doubleBufferPtr, &doubleBuffer);
  if ((squarePtr->gc == None) && (doubleBuffer)) {
    XGCValues gcValues;
    gcValues.function = GXcopy;
    gcValues.graphics_exposures = False;
    squarePtr->gc = Tk_GetGC(squarePtr->tkwin,
			     GCFunction|GCGraphicsExposures, &gcValues);
  }

  /*
   * Register the desired geometry for the window. Then arrange for the
   * window to be redisplayed.
   */

  Tk_GeometryRequest(squarePtr->tkwin, 200, 150);
  Tk_GetPixelsFromObj(NULL, squarePtr->tkwin, squarePtr->borderWidthPtr,
		      &borderWidth);
  Tk_SetInternalBorder(squarePtr->tkwin, borderWidth);
  if (!squarePtr->updatePending) {
    Tcl_DoWhenIdle(SquareDisplay, squarePtr);
    squarePtr->updatePending = 1;
  }
  KeepInWindow(squarePtr);
  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * SquareObjEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various events on
 *	squares.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get cleaned up. When
 *	it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
SquareObjEventProc(
		   ClientData clientData,	/* Information about window. */
		   XEvent *eventPtr)		/* Information about event. */
{
  Square *squarePtr = clientData;

  if (eventPtr->type == Expose) {
    if (!squarePtr->updatePending) {
      Tcl_DoWhenIdle(SquareDisplay, squarePtr);
      squarePtr->updatePending = 1;
    }
  } else if (eventPtr->type == ConfigureNotify) {
    KeepInWindow(squarePtr);
    if (!squarePtr->updatePending) {
      Tcl_DoWhenIdle(SquareDisplay, squarePtr);
      squarePtr->updatePending = 1;
    }
  } else if (eventPtr->type == DestroyNotify) {
    if (squarePtr->tkwin != NULL) {
      Tk_FreeConfigOptions((char *) squarePtr, squarePtr->optionTable,
			   squarePtr->tkwin);
      if (squarePtr->gc != None) {
	Tk_FreeGC(squarePtr->display, squarePtr->gc);
      }
      squarePtr->tkwin = NULL;
      Tcl_DeleteCommandFromToken(squarePtr->interp,
				 squarePtr->widgetCmd);
    }
    if (squarePtr->updatePending) {
      Tcl_CancelIdleCall(SquareDisplay, squarePtr);
    }
    Tcl_EventuallyFree(squarePtr, SquareDestroy);
  }
}

/*
 *----------------------------------------------------------------------
 *
 * SquareDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted. If the
 *	widget isn't already in the process of being destroyed, this command
 *	destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */

static void
SquareDeletedProc(
		  ClientData clientData)	/* Pointer to widget record for widget. */
{
  Square *squarePtr = clientData;
  Tk_Window tkwin = squarePtr->tkwin;

  /*
   * This procedure could be invoked either because the window was destroyed
   * and the command was then deleted (in which case tkwin is NULL) or
   * because the command was deleted, and then this procedure destroys the
   * widget.
   */

  if (tkwin != NULL) {
    Tk_DestroyWindow(tkwin);
  }
}

/*
 *--------------------------------------------------------------
 *
 * SquareDisplay --
 *
 *	This procedure redraws the contents of a square window. It is invoked
 *	as a do-when-idle handler, so it only runs when there's nothing else
 *	for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

static void
SquareDisplay(ClientData clientData) 
{
  Square *squarePtr = clientData;
  Tk_Window tkwin = squarePtr->tkwin;
  Display *display = squarePtr->display;
  Pixmap pm = None;
  Drawable d;
  int borderWidth, size, relief;
  Tk_3DBorder bgBorder, fgBorder;
  int doubleBuffer;
  Tk_Font tkfont = squarePtr->tkfont;

  squarePtr->updatePending = 0;
  if (! Tk_IsMapped(tkwin)) {
    return;
  }

  /*
   * Create a pixmap for double-buffering, if necessary.
   */

  MyFontInfoSq(squarePtr);

  Tcl_GetBooleanFromObj(NULL, squarePtr->doubleBufferPtr, &doubleBuffer);
  if (doubleBuffer) {
    pm = Tk_GetPixmap(Tk_Display(tkwin), Tk_WindowId(tkwin), Tk_Width(tkwin), Tk_Height(tkwin), DefaultDepthOfScreen(Tk_Screen(tkwin)));
    d = pm;
  } else {
    d = Tk_WindowId(tkwin);
  }
    
  /*
   * Redraw the widget's background and border.
   */

  Tk_GetPixelsFromObj(NULL, squarePtr->tkwin, squarePtr->borderWidthPtr,
		      &borderWidth);
  bgBorder = Tk_Get3DBorderFromObj(squarePtr->tkwin,squarePtr->bgBorderPtr);
  Tk_GetReliefFromObj(NULL, squarePtr->reliefPtr, &relief);
  Tk_Fill3DRectangle(tkwin, d, bgBorder, 0, 0, Tk_Width(tkwin), Tk_Height(tkwin), borderWidth, relief);
    
  /*
   * Display the square.
   */
    
  Tk_GetPixelsFromObj(NULL, squarePtr->tkwin, squarePtr->sizeObjPtr, &size);
  fgBorder = Tk_Get3DBorderFromObj(squarePtr->tkwin,squarePtr->fgBorderPtr);
  Tk_Fill3DRectangle(tkwin, d, fgBorder, squarePtr->x, squarePtr->y, size, size, borderWidth, TK_RELIEF_RAISED);

  XGCValues *newgc;
  //GC newgc;
  XGCValues gcValues;
  gcValues.function = GXcopy;
  gcValues.graphics_exposures = False;
  gcValues.font = Tk_FontId(squarePtr->tkfont);
  gcValues.foreground = Tk_GetColor(squarePtr->interp, tkwin, "white")->pixel;
  gcValues.background = Tk_GetColor(squarePtr->interp, tkwin, "black")->pixel;
  newgc = XCreateGC(display, d, GCFunction|GCGraphicsExposures|GCFont|GCForeground|GCBackground, &gcValues);
  Tk_DrawImageString(Tk_Display(tkwin),d,tkfont,newgc,0,0,"Z",1);

  gcValues.background = Tk_GetColor(squarePtr->interp, tkwin, "blue")->pixel;
  XChangeGC(display, newgc, GCBackground, &gcValues);
  squareDrawImageString(squarePtr,Tk_Display(tkwin),d,newgc,15,0,"i",1);
  
  gcValues.background = Tk_GetColor(squarePtr->interp, tkwin, "red")->pixel;
  XChangeGC(display, newgc, GCBackground, &gcValues);
  squareDrawImageString(squarePtr,Tk_Display(tkwin),d,newgc,30,0,"Q",1);

  /*
   * If double-buffered, copy to the screen and release the pixmap.
   */

  if (doubleBuffer) {
    XCopyArea(Tk_Display(tkwin), pm, Tk_WindowId(tkwin), squarePtr->gc,
	      0, 0, (unsigned) Tk_Width(tkwin), (unsigned) Tk_Height(tkwin),
	      0, 0);
    Tk_FreePixmap(Tk_Display(tkwin), pm);
  }

  Tcl_SetObjResult(squarePtr->interp,Tcl_NewStringObj(Tcl_GetStringResult(squarePtr->interp), -1));
}

/*
 *----------------------------------------------------------------------
 *
 * SquareDestroy --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release to
 *	clean up the internal structure of a square at a safe time (when
 *	no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the square is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
SquareDestroy(
	      char *memPtr)		/* Info about square widget. */
{
  Square *squarePtr = (Square *) memPtr;

  ckfree((char *) squarePtr);
}

/*
 *----------------------------------------------------------------------
 *
 * KeepInWindow --
 *
 *	Adjust the position of the square if necessary to keep it in the
 *	widget's window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The x and y position of the square are adjusted if necessary to keep
 *	the square in the window.
 *
 *----------------------------------------------------------------------
 */

static void
KeepInWindow(
	     register Square *squarePtr)	/* Pointer to widget record. */
{
  int i, bd, relief;
  int borderWidth, size;

  Tk_GetPixelsFromObj(NULL, squarePtr->tkwin, squarePtr->borderWidthPtr,
		      &borderWidth);
  Tk_GetPixelsFromObj(NULL, squarePtr->tkwin, squarePtr->xPtr,
		      &squarePtr->x);
  Tk_GetPixelsFromObj(NULL, squarePtr->tkwin, squarePtr->yPtr,
		      &squarePtr->y);
  Tk_GetPixelsFromObj(NULL, squarePtr->tkwin, squarePtr->sizeObjPtr, &size);
  Tk_GetReliefFromObj(NULL, squarePtr->reliefPtr, &relief);
  bd = 0;
  if (relief != TK_RELIEF_FLAT) {
    bd = borderWidth;
  }
  i = (Tk_Width(squarePtr->tkwin) - bd) - (squarePtr->x + size);
  if (i < 0) {
    squarePtr->x += i;
  }
  i = (Tk_Height(squarePtr->tkwin) - bd) - (squarePtr->y + size);
  if (i < 0) {
    squarePtr->y += i;
  }
  if (squarePtr->x < bd) {
    squarePtr->x = bd;
  }
  if (squarePtr->y < bd) {
    squarePtr->y = bd;
  }
}


int __declspec(dllexport) Square_Init(Tcl_Interp *interp) {
  if (Tcl_InitStubs(interp, "8.4" , 0) == NULL) {
    return TCL_ERROR;
  }
  if (Tk_InitStubs(interp, "8.0" , 0) == NULL) {
    return TCL_ERROR;
  }
    
  Tcl_CreateObjCommand(interp, "Square", SquareObjCmd, (ClientData) Tk_MainWindow(interp), (Tcl_CmdDeleteProc *)NULL);

  Tcl_PkgProvide(interp,"Square","0.1");

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * squarePtrDrawImageString --
 *	Draw a character with a given color, and
 *      draws the underlying background of the bbox 
 *      containing those characters with a given 
 *      color.
 *
 * Results:
 *      None. 
 *
 * Side effects:
 *      None.      
 *
 *--------------------------------------------------------------
 */
void
squareDrawImageString(Square *squarePtr,
		      Display *display,
		      Drawable d,
		      GC gc,
		      int x, int y, 
		      char *string,
		      int length) 
{    
  if (d == None) {
    return;
  }
    
  HDC hdc;
  HFONT oldFont;
  LPTSTR lpchText;
  WinFont *fontPtr;
  SubFont *lastSubFontPtr;
  TkWinDCState state;
  Tk_Font tkfont = squarePtr->tkfont; /* Corrext , checked by issuing name and metrics */

  lpchText = string;

  fontPtr = (WinFont *) tkfont;
  //fontPtr = (WinFont *) gc->font;
  lastSubFontPtr = &(fontPtr->subFontArray[0]);
  //display->request++;
  //hdc = GetDC(TkWinGetHWND(Tk_WindowId(squarePtr->tkwin)));
  hdc = TkWinGetDrawableDC(display, d, &state);
  SetROP2(hdc, SRCCOPY);

  SetBkMode(hdc, OPAQUE);
  SetBkColor(hdc, gc->background);
  SetTextColor(hdc, gc->foreground);

  oldFont = SelectObject(hdc, lastSubFontPtr->hFont0);
  
  TextOut(hdc, x, y, lpchText, 1);

  SelectObject(hdc, oldFont);
  //ReleaseDC(fontPtr->hwnd, hdc);
  TkWinReleaseDrawableDC(d, hdc, &state);

  return;
}

/*
 *--------------------------------------------------------------
 *
 * MyFontInfoSq --
 *      Gives font information. Only use non-Xft fonts
 *
 * Results:
 *      None. 
 *
 * Side effects:
 *      None.      
 *
 *--------------------------------------------------------------
 */
void
MyFontInfoSq(Square *squarePtr) 
{
  Tk_FontMetrics fm;
  Tk_Font tkfont = squarePtr->tkfont;

  Tk_GetFontMetrics(tkfont, &fm);
  squarePtr->charWidth = Tk_TextWidth(tkfont,"0", 1);
  squarePtr->charHeight  = fm.ascent+fm.descent;
  squarePtr->ascent = fm.ascent;
  squarePtr->descent = fm.descent;

  fprintf(stderr, "%s %d %d\n", Tk_NameOfFont(squarePtr->tkfont), squarePtr->charWidth, squarePtr->charHeight);
  fflush(stderr);

  return;
}



