/* polyreduce.c, procedures to 
   reduce the number of polygons (triangles)
   in a triangle mesh, generated by the 
   'marching cubes' algorithm... 
   
   Jesper James Jensen 1995  */

/* les includo filez... */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "polyr.h"
#include <tcl.h>

/* some global defines */
#define VERTHASHSIZE 19991
#define EDGEHASHSIZE 39991
#define MAXNUMADJS 20
#define TRUE 1
#define FALSE 0

/* do more self checking */
#define CAUTIOUS 

/*  first some relevant structures:  */

/* The extern vertice list w're processing */
extern float * VERTICES;
extern int NUM_VERTICES;

/* globals */
static suc[4] = {1, 2, 0, 1};    
float edgelength;

/* a vertice structure */
typedef struct Vert {
  struct Vert *prev,*next;        /* w're using lists here! */
  struct Vert *hnext;             /* for hashing */
  float x,y,z;                    /* Coordinates */
  float nx,ny,nz;                 /* Normal Vector */
  short int num_adjv;             /* Number of adjacent vertices */
  struct Vert *adjv[MAXNUMADJS];  /* Pointer to adjacent vertices */
  short int num_adjt;             /* Number of adjacent triangles */
  struct Tri *adjt[MAXNUMADJS];   /* Pointer to adjacent triangles */
  int index;                      /* index number used for edge hashing */
  short int edge;                 /* is this vertex on an edge */
  short int donottouch;           /* this triangle is in a tunnel */
} Vert;

/* A triangle structure */
typedef struct Tri {
  struct Tri *prev,*next;         /* again prepare for lists */
  struct Tri *hnext;              /* again hashing */
  short int adjcount;             /* number of adjacent triangles */
  struct Tri *(adj[3]);           /* adjacent triangles */
  Vert *vert[3];                  /* Pointer to the three vertices */  
  float edgelength[3];            /* Length of the edges */
  float nx,ny,nz;                 /* Normal vector of the triangle */
  short int objectnr;             /* which object this triangle belongs to */
  short int removable;            /* flag to indicate which list th tri. is in now */
} Tri;

/* Edge struct used for edge hashing */
typedef struct Edge {
    struct Edge *hnext;
    struct Tri *tri;
    int index0, index1;
} Edge;

/* list of vertices */
typedef struct Vertlist {        
  Vert *first, *last;
} Vertlist;

/* list of triangles */
typedef struct Trilist {         
    Tri *first,*last;
} Trilist;

/* handle for the splitted objects */
typedef struct triobject {
  Tri *tri;
  int numtris;
} triobject;

/* the global lists */
static Vert **verthashlist;
static Edge **edgehashlist;
static Edge *edgearray;
static Edge *freeedges;
static Vertlist *vertlist,*donevert;  /* The global vertex list + trash list */
static Trilist *trilist,*donetri;     /* The global triangle list + trash list */
static Trilist *removable;            /* List of trinagles that can be removed */

Tcl_Obj 
*dump(Tcl_Interp *interp, int vcount, int tcount, int offset, float xxm[2][3]);
Tcl_Obj 
*reduce(Tcl_Interp *interp, int offset, float xxm[2][3]);


/* init vertex hash list */
void 
hashvertbegin() {       
  int i;
  
  verthashlist = (Vert**)malloc(VERTHASHSIZE*sizeof(Vert*));
  for(i=0; i<VERTHASHSIZE; i++) 
    verthashlist[i] = 0;
}


/* calc. the hash value for a given vertex */
int 
hashvert(vert)
Vert *vert;
{
  int *buf = (int *)&vert->x;
  long hash = 0;
  
  /* hashing equal to that of 'compact.c' for consistency ;-) */
  hash = (((buf[0]*283+buf[1])*283)+buf[2]);
  hash &= 0x7fffffff;

  return (int)(hash%VERTHASHSIZE);
}


/* insert a vertex in the vertex hash list */
Vert *
hashvertadd(vert)
Vert *vert;
{
    int pos;
    Vert *vptr;
#define vequal(p,q) (p->x==q->x && p->y==q->y && p->z==q->z)
    pos = hashvert(vert);

    /* check if already present */
    for(vptr = verthashlist[pos]; vptr; vptr = vptr->hnext)
	if(vequal(vert,vptr)) return vptr;
    vert->hnext = verthashlist[pos];
    verthashlist[pos] = vert;

    return 0;
}


/* Free the vertex hash list */
hashvertend()
{
    free(verthashlist);
}

/* Init the edge hash list */
hashedgebegin(int npolys)
{
    int i;

    edgehashlist = (Edge**)malloc(EDGEHASHSIZE*sizeof(Edge*));
    edgearray = freeedges = (Edge*)malloc(3*npolys*sizeof(Edge));
    for(i=0; i<EDGEHASHSIZE; i++) 
	edgehashlist[i] = 0;
}


/* Calc. the hash value for en edge (two vertices) */
hashedge(index0,index1)
int index0, index1;
{
    long val;

    val = index0+index1;
    val = val&0x7fffffff;
    return (int)(val%EDGEHASHSIZE);
}


/* find i entry in the edge hash list */
Edge *
hashedgefind(v0,v1)
Vert *v0, *v1;
{
    return edgehashlist[hashedge(v0->index,v1->index)];
}


/* insert edge in hash list */
hashedgeadd(tri,v0,v1)
Tri *tri;
Vert *v0, *v1;
{
    int pos;
    int index0, index1;
    Edge *edge;

    index0 = v0->index;
    index1 = v1->index;
    pos = hashedge(index0,index1);
    edge = freeedges++;
    edge->index0 = index0;
    edge->index1 = index1;
    edge->tri = tri;
    edge->hnext = edgehashlist[pos];
    edgehashlist[pos] = edge;
}


/* end edge hashing */
hashedgeend()
{
    free(edgehashlist);
    free(edgearray);
}


/* Remove a triangle from the adj. lists of connected triangles */
removeadjacencies(tri)
Tri *tri;
{
  register i,j;
  Tri *adjtri;
  
  for (i=0; i<3; i++) {
    adjtri = tri->adj[i];
    if (adjtri) {
      adjtri->adjcount -= 1;
      for (j=0; j<3; j++) {
	if (tri == adjtri->adj[j]) {
	  adjtri->adj[j] = 0;
	  break;
	}
      }
    }
  }
}


/* Print information about a triangle */
void
printtri(Tri *t) {
  int i;
  printf("Tri: %x, Vertices: [%d,%d,%d], Adjs[%d]:",
	 t,(t->vert[0])->index,(t->vert[1])->index,(t->vert[2])->index,t->adjcount);
/*  printf("Normal: %f %f %f\n",t->nx,t->ny,t->nz); */
  for(i=0; i<3; i++)
    printf("%x ",t->adj[i]);
  printf("\n"); 
}


/* Print information about a vertex */
void
printvert(Vert *v) {
  int i;

  printf("Vert: %3d, [%2d] :",v->index,v->num_adjv);
  for(i=0; i<v->num_adjv; i++)
    printf("%d ",(v->adjv[i])->index);
  printf("\n           [%2d] :",v->num_adjt);
  for(i=0; i<v->num_adjt; i++)
    printf("%x[%d] ",(v->adjt[i]),(v->adjt[i])->adjcount);
  printf("\n");
}



/* allocate space for and initialize a vert list */
Vertlist *
makevertlist() 
{
  Vertlist *tmp;
  
  if ((tmp=(Vertlist*)(malloc(sizeof(Vertlist))))==0) {
    fprintf(stderr,"makevertlist: out of memory.  abort.\n");
    exit(1);
  }
  tmp->first = tmp->last = 0;
  return tmp;
}


/* allocate a new vertex */
void
insertvert(list,item)
     Vertlist *list;
     Vert *item;
{
  /* insert the new item at the top of the list */
  Vert *tmp;
  
  if (list->first) {
    tmp = list->first;
    list->first = item;
    item->next = tmp;
    item->prev = 0;
    tmp->prev = item;
  } else {
    list->first = list->last = item;
    item->prev = item->next = 0;
  }
}


/* delete a vertex from the list */
void
deletevert(list,item)
     Vertlist *list;
     Vert *item;
{
  /* delete the item from the list */
  if (list->first == item) {
    if (list->last == item) {
      /* this is the only item in the list */
      list->first = list->last = 0;
    } else {
      /* this is the first item in the list */
      list->first = item->next;
      list->first->prev = 0;
    }
  } else if (list->last == item) {
    /* this is the last item in the list */
    list->last = item->prev;
    list->last->next = 0;
  } else {
    item->prev->next = item->next;
    item->next->prev = item->prev;
  }
  item->prev = item->next = 0;
}


/* allocate space for and initialize a tri list */
Trilist *
maketrilist() 
{
  Trilist *tmp;
  
  if ((tmp=(Trilist*)(malloc(sizeof(Trilist))))==0) {
    fprintf(stderr,"maketrilist: out of memory.  abort.\n");
    exit(1);
  }
  tmp->first = tmp->last = 0;
  return tmp;
}


/* Insert a triangle in the triangle list */
void
inserttri(list,item)
     Trilist *list;
     Tri *item;
{
  /* insert the new item at the top of the list */
  Tri *tmp;
  
  if (list->first) {
    tmp = list->first;
    list->first = item;
    item->next = tmp;
    item->prev = 0;
    tmp->prev = item;
  } else {
    list->first = list->last = item;
    item->prev = item->next = 0;
  }
}


/* Delete a triangle from the list */
void 
deletetri(list,item)
     Trilist *list;
     Tri *item;
{
  /* delete the item from the list */
  if (list->first == item) {
    if (list->last == item) {
      /* this is the only item in the list */
      list->first = list->last = 0;
    } else {
      /* this is the first item in the list */
      list->first = item->next;
      list->first->prev = 0;
    }
  } else if (list->last == item) {
    /* this is the last item in the list */
    list->last = item->prev;
    list->last->next = 0;
  } else {
    item->prev->next = item->next;
    item->next->prev = item->prev;
  }
  item->prev = item->next = 0;
}


/* Allocate a new vertex */
Vert *
makevert() {
  Vert *tmp;
  
  /* using ordinary malloc */
  if (!(tmp=(Vert *) malloc(sizeof(Vert)))) {
    fprintf(stderr,"makevert: out of memory.\n");
    exit(1);
  }
  tmp->prev = tmp->next = 0;
  tmp->num_adjv = tmp->num_adjt = 0;
  tmp->edge=0;
  tmp->donottouch=0;
  
  return tmp;
}


/* Kill a vertex */
void 
killvert(Vert *v) {
  deletevert(vertlist,v);
  free(v);
}


/* Kill a triangle */
void
killtri(Tri *t) {
  if (t->removable)
    deletetri(removable,t);
  else
    deletetri(trilist,t);
  free(t);
}


/* Allocate a new triangle */
Tri *
maketri() {
  int i;
  Tri *tmp;
  
  /* using ordinary malloc */
  if (!(tmp=(Tri *) malloc(sizeof(Tri)))) {
    fprintf(stderr,"maketri: out of memory.\n");
    exit(1);
  }
  tmp->prev = tmp->next = 0;
  tmp->adjcount = 0;
  for (i=0; i<3; i++) {
    tmp->adj[i] = 0;
    tmp->vert[i] = 0;
  }
  tmp->objectnr=-1;
  tmp->removable=0;
  return tmp;
}


/* return TRUE if vert is one of the vertices in tri, otherwise FALSE */
int 
ismember(vert,tri)
     Vert *vert;
     Tri *tri;
{
  return (vert==tri->vert[0]) | (vert==tri->vert[1]) | (vert==tri->vert[2]);
}


/* returns the index of the vertex in tri that is not in tri2 */
int 
notcommon(tri,tri2)
     Tri *tri,*tri2;
{
  int i;
  
  for (i=0; i<3; i++)
    if (!ismember(tri->vert[i],tri2))
      return i;
  return -1;
}


/* returns vert's index in tri */
int
getvertindex(Tri *tri, Vert *vert)
{
  if (tri->vert[0]==vert) 
    return 0;
  if (tri->vert[1]==vert) 
    return 1;
  if (tri->vert[2]==vert)
    return 2;
  
  return -1;
}


/* Add a triangle to the adjt list of vertex v */
void
vertaddtri(Vert* v, Tri *t) 
{
  if (v->num_adjt==MAXNUMADJS-1) {
    fprintf(stderr,"%s: ERROR vertaddtri, to many adj. vertices (%d)\n",MY_NAME,v->index);
    printf("Vertex coordinates: [%f %f %f]\n",v->x,v->y,v->z);
    printf("This can happen if the vertex is excactly on a grid point\n");
    printf("Try increasing MAXNUMADJS\n");
    exit(1);
  }
#ifdef CAUTIOUS
  if (istriinvert(v,t)) {
    fprintf(stderr,"%s: ERROR vertaddtri, adding same tri (%d) to verts adjt list %x \n"
	    ,MY_NAME,t,v->index);
    printvert(v);
    exit(1);
  }
#endif
  v->adjt[v->num_adjt]=t;
  v->num_adjt+=1;
}


/* Add a vertex to the adjv list of vertex v */
void
vertaddvert(Vert* v, Vert *new) 
{
  if (v->num_adjv==MAXNUMADJS-1) {
    fprintf(stderr,"%s: ERROR vertaddvert, to many adj. tris %d %d %d\n",
	    MY_NAME,v->index,v->num_adjt,v->num_adjv);
    printf("Vertex coordinates: [%f %f %f]\n",v->x,v->y,v->z);
    printf("This can happen if the vertex is excactly on a grid point\n");
    printf("Try increasing MAXNUMADJS\n");
    exit(1);
  }
#ifdef CATIOUS
  if (v==new) {
    printf("\n%s: ERROR vertadderror, adding vert to itself\n",MY_NAME);
    exit(1);
  }
  if (istriinvert(v,new)) {
    fprintf(stderr,"%s: ERROR vertaddvert, adding same vert (%x) to verts adjt list %x \n"
	    ,MY_NAME,v->index,new->index);
    printvert(v);
    exit(1);
  }
#endif
  v->adjv[v->num_adjv]=new;
  v->num_adjv+=1;
}


/* checks if vertex 'check' is in the adjv list of vertex v */
int
isvertinvert(Vert *v, Vert *check)
{
  int i;
  for(i=0;i<v->num_adjv; i++)
    if (v->adjv[i]==check)
      return 1;
  return 0;
}


/* checks if triangle 'check' is in the adjv list of vertex v */
int
istriinvert(Vert *v, Tri *check)
{
  int i;
  for(i=0;i<v->num_adjt; i++)
    if (v->adjt[i]==check)
      return 1;
  return 0;
}


/* returns the number of verts v1 and v2 has in common */
int
vertvertincommon(Vert *v1, Vert *v2) {
  int i,count=0;
  
  /* a quick hack ! */
  for(i=0; i<v1->num_adjv; i++)
    count+=isvertinvert(v2,(v1->adjv[i]));
  
  return count;
}


/* checks the number of times 'check' is in the adjv list of vertex v */
int
numvertinvert(Vert *v, Vert *check)
{
  int i,count=0;
  for(i=0;i<v->num_adjv; i++)
    if (v->adjv[i]==check)
      count++;
  return count;
}


/* removes the vertex 'old' from the adjv list of vertex v */
void
vertremovevert(Vert *v, Vert *old, int a) {
  int i,found=0;
  
  for(i=0; i<v->num_adjv; i++) {
    if (found)
      v->adjv[i-found]=v->adjv[i];
    if (v->adjv[i]==old) {
      found=+1;
      continue;
    }
  }
#ifdef CAUTIOUS
  if (found!=1) {
    printf("vertremovert: error in ajd. generation %d %d [%d] \n", v->index,old->index,a);
    exit(1);
  }
#endif
  v->num_adjv-=found;
}


/* removes the triangle 'old' from the adjt list of vertex v */
void
vertremovetri(Vert *v, Tri *old) {
  int i,found=0;
  
  for(i=0; i<v->num_adjt; i++) {
    if (found)
      v->adjt[i-found]=v->adjt[i];
    if (v->adjt[i]==old) {
      found=+1;
      continue;
    }
  }
#ifdef CAUTIOUS
  if (found!=1) {
    printf("vertremovetri: error in ajd. generation %d %x\n", found, v->index, old);
    exit(1);
  }
#endif
  v->num_adjt-=1;
}


/* changes references from vertex 'old' to 'new' in adjv list for vertex v */
void
vertupdateadjvert(Vert *v, Vert *old, Vert *new) {
  int i;

  for(i=0; i<v->num_adjv; i++) {
    if (v->adjv[i]==old) {
      v->adjv[i]=new;
      break;
    }
  }
}


/* changes references from triangle 'old' to 'new' in adjt list for vertex v */
void
vertupdateadjtri(Vert *v, Tri *old, Tri *new) {
  int i;
  
  for(i=0; i<v->num_adjt; i++) {
    if (v->adjt[i]==old) {
      v->adjt[i]=new;
      break;
    }
  }
}


/* changes references to triangle 'old' to 'new' in adj list for triangle t */
void
triupdateadjtri(Tri *t, Tri *old, Tri *new) {
  int i;
  
  for(i=0; i<3; i++) {
    if (t->adj[i]==old) {
      t->adj[i]=new;
      break;
    }
  }
}


/* changes the reference from vertex 'old' to 'new' in triangle t */
triupdatevert(Tri *t, Vert *old, Vert *new) {
  int i;
  
  for(i=0; i<3; i++) {
    if (t->vert[i]==old) {
      t->vert[i]=new;
      break;
    }
  }
}


/* returns the squared distance between the two vertices */
float
vertdist(Vert *v1, Vert *v2) {
#define SQ(a) ((a)*(a))  
  return (SQ(v1->x-v2->x)+SQ(v1->y-v2->y)+SQ(v1->z-v2->z));
}


/* calculates the squared lenght of all sides for a triangle */
void
calctriedges(Tri *tri) {
  Vert *v0,*v1,*v2;

  v0=tri->vert[0];
  v1=tri->vert[1];
  v2=tri->vert[2];

  tri->edgelength[0]=SQ(v0->x-v1->x)+SQ(v0->y-v1->y)+SQ(v0->z-v1->z);
  tri->edgelength[1]=SQ(v1->x-v2->x)+SQ(v1->y-v2->y)+SQ(v1->z-v2->z);
  tri->edgelength[2]=SQ(v2->x-v0->x)+SQ(v2->y-v0->y)+SQ(v2->z-v0->z);
}


/* move triangles between trilist and removable lists */
void
checktri(Tri *tri, float maxedgesize) {
  if ((tri->edgelength[0]<maxedgesize) ||
      (tri->edgelength[1]<maxedgesize) ||
      (tri->edgelength[2]<maxedgesize))
  {
    /* tri belongs in removable list */
    if (!tri->removable) {
      tri->removable=1;
      deletetri(trilist,tri);
      inserttri(removable,tri);
    }
  } else {
    /* tri belongs in trilist */
    if (tri->removable) {
      tri->removable=0;
      deletetri(removable,tri);
      inserttri(trilist,tri);
    }    
  }
}

    
/* move a triangle from removable list to trilist */
void
forcemovetri(Tri *tri) {
  tri->removable=0;
  deletetri(removable,tri);
  inserttri(trilist,tri);
}

     
/* move all references from vert 'src' to vert 'dest' */
void
movevertrefs(Vert *src, Vert *dest) {
  int i,j;

  /* fix the vertex pointers */
  for(i=0; i<src->num_adjv; i++) {
    if (isvertinvert(src->adjv[i],src)) 
      vertupdateadjvert(src->adjv[i],src,dest);
    if (!isvertinvert(dest,src->adjv[i]))
      vertaddvert(dest,src->adjv[i]);
  }
  /* fix the tri pointers */
  for(i=0; i<src->num_adjt; i++) {
#ifdef CAUTIOUS
    if (getvertindex(src->adjt[i],dest)!=-1) 
	printf("ERROR: movevertrefs, trying illegal move! %x %d->%d\n",
	       src->adjt[i],src->index,dest->index);
#endif
    triupdatevert(src->adjt[i],src,dest);
    if (!istriinvert(dest,src->adjt[i]))
      vertaddtri(dest,src->adjt[i]);
  }
}


/* when killing a triangles the adj. triangles needs updated adj. info */
void
exchangeadjacencies(Tri *t, int idx) {
  int i;
  static pre[3] = {2, 0, 1};
  Tri *adj1,*adj2;
  
  adj1=t->adj[idx];
  adj2=t->adj[pre[idx]];
  
  for (i=0; i<3; i++) {
    if (t == adj1->adj[i]) {
      adj1->adj[i]=adj2;
      break;
    }
  }
  for (i=0; i<3; i++) {
    if (t == adj2->adj[i]) {
      adj2->adj[i]=adj1;
      break;
      }
  }
}


/* Calculate the normal for a triangle */
void
calc_tri_normal(Tri *t)
{
    float u[3], v[3];
    float *p1,*p2,*p3,*n;
    float *v1,*v2,*v3;
    float sum, mag;

    p1=&(t->vert[0])->x;
    p2=&(t->vert[1])->x;
    p3=&(t->vert[2])->x;
    v1=&(t->vert[0])->nx;
    v2=&(t->vert[1])->nx;
    v3=&(t->vert[2])->nx;
    n=&t->nx;

    u[0] = p3[0] - p2[0];
    u[1] = p3[1] - p2[1];
    u[2] = p3[2] - p2[2];
    v[0] = p1[0] - p2[0];
    v[1] = p1[1] - p2[1];
    v[2] = p1[2] - p2[2];
    n[0] = u[1] * v[2] - u[2] * v[1];
    n[1] = u[2] * v[0] - u[0] * v[2];
    n[2] = u[0] * v[1] - u[1] * v[0];

    sum = n[0] * n[0] + n[1] * n[1] + n[2] * n[2];
    mag = (float) sqrt((double) sum);
    if (mag == 0.0)
	mag = 1.0;

    n[0] = n[0] / mag;
    n[1] = n[1] / mag;
    n[2] = n[2] / mag;

    v1[0] = v2[0] = v3[0] = n[0];
    v1[1] = v2[1] = v3[1] = n[1];
    v1[2] = v2[2] = v3[2] = n[2];
}


/* recursively walk through mesh, numbering all triangles connected
   to tri with objnr */
void
triupdateobjectnr(Tri *tri, int objnr) {
  int i,j;
  
  if (tri->objectnr==-1) {
    tri->objectnr=objnr;
    for (i=0; i<3; i++)
      for (j=0; j<(tri->vert[i])->num_adjt; j++) 
	if ((tri->vert[i])->adjt[j])
	  triupdateobjectnr((tri->vert[i])->adjt[j],objnr);
  }
}


/* recursively walk through mesh, setting all triangles conected
   to tri to objnr=-1 for later removal */
void
triremovesmall(Tri *tri) {
  int i,j;
  
  tri->objectnr=-1;
  for (i=0; i<3; i++)    
    for (j=0; j<(tri->vert[i])->num_adjt; j++) 
      if (((tri->vert[i])->adjt[j])->objectnr!=-1)
	triremovesmall((tri->vert[i])->adjt[j]);
}


/* seperates the global triangle mesh into seperate internally 
   connected objects (triangle meshes) */
void 
triseperate() {
  int i,count=0,newcount=0;
  Tri *tri,*nexttri;
  Vert *vert,*nextvert;
  struct triobject *tris, *newtris;
  float *mask;

  /* init the trash lists */
  donevert=makevertlist();
  donetri=maketrilist();

  /* give all triangles a number indicating their conectivity */
  for(tri=trilist->first; tri;) {
    if (tri->objectnr==-1) {
      triupdateobjectnr(tri,count);
      count++;
    }    
    tri=tri->next;
  }
  /* make a list of seed triangles and number of tris in the seperate objects */
  tris=(struct triobject *)calloc(count,sizeof(struct triobject));
  newtris=(struct triobject *)calloc(count,sizeof(struct triobject));
  for(tri=trilist->first; tri;) {
    if (!tris[tri->objectnr].tri)
      tris[tri->objectnr].tri=tri;
    tris[tri->objectnr].numtris++;
    tri=tri->next;
  }

  /* mark all objects with under 'MANTRIS' triangles for killing */
  if (VERBOSE)
    printf("%s: Removing volumes with under %d triangles\n",MY_NAME,MAXPOLS);
  for(i=0; i<count; i++) {
    if (tris[i].numtris>MAXPOLS) {
      newtris[newcount].tri=tris[i].tri;
      newtris[newcount].numtris=tris[i].numtris;
      newcount++;
    }
    else
      triremovesmall(tris[i].tri);
  }

  if (VERBOSE)
    printf("%s: object consists of %d volume%s (%d killed)\n",
	   MY_NAME,newcount,(newcount==1) ? "":"s",count-newcount);

  /* transfer all the killed vertices and triangles to the trash lists */
  for(tri=trilist->first; tri;) {
    nexttri=tri->next;
    if (tri->objectnr==-1) {
      for(i=0; i<3; i++)
	if ((tri->vert[i])->index!=-1) {
	  (tri->vert[i])->index=-1;
	  deletevert(vertlist,tri->vert[i]);
	  insertvert(donevert,tri->vert[i]);
	}
      deletetri(trilist,tri);
      inserttri(donetri,tri);
    }
    tri=nexttri;
  }
  
  /* kill the trash vertices */
  count=0;
  for(vert=donevert->first; vert;) {
    nextvert=vert->next;
    deletevert(donevert,vert);
    free(vert);
    vert=nextvert;
    count++;
  }
  free(donevert);

  /* kill the trash triangles */
  newcount=0;
  for(tri=donetri->first; tri;) {
    nexttri=tri->next;
    deletetri(donetri,tri);
    free(tri);
    tri=nexttri;
    newcount++;
  }
  if (VERBOSE)
    if(count || newcount)
      printf("%s: Killed volumes had %d vertices and %d triangles\n",
	     MY_NAME,count,newcount);
  
  /* done - cleanup */
  free(tris);
  free(newtris);
  free(donetri);
}


/* Kill the triangle t1 by joining vertices v1 and v2 */
int
collapsetri(Tri *t1, Vert *v1, Vert *v2, float edgesize) {
  int i,j,edge;
  float newxyz[3];

  Vert *notcommon1=NULL, *notcommon2=NULL;
  Tri *t2,*adj1,*adj2,*adj3;

  /* if w're on an edge, do nothing */
  if ((v1->edge) || (v2->edge))
    return -1;
  
  /* is there a loop problem? */
  if ((v1->donottouch) || (v2->donottouch))
    return -1;

#ifdef CAUTIOUS
  if (v1==v2) {
    printf("ERROR calling collapsetri with v1=v2 (t=%x) \n",t1);
    return -1;
  }
#endif

  /* all right, first find the two triangles common to v1 & v2 */
  if (getvertindex(t1,v1)==-1) {
    printf("v1 not in t1 %x\n",t1);
    return -1;
  }

#ifdef CAUTIOUS
  if (t1->vert[0]==t1->vert[1] ||
      t1->vert[1]==t1->vert[2] ||
      t1->vert[2]==t1->vert[0])
    printf("ERROR, the triangle %x is degenerate\n",t1);
#endif

  t2=t1->adj[getvertindex(t1, v1)];
  
#ifdef CAUTIOUS
  if (getvertindex(t2,v1)==-1 || getvertindex(t2,v2)==-1)
    printf("ERROR, in mesh generation\n");
#endif

  /* first update triangle adjancies: find the other adjacent triangles */
  i=notcommon(t1,t2);
  j=notcommon(t2,t1);
  
  /* do not to degenerate the vertices (always>=4 adjt's) */
  if (((t1->vert[i])->num_adjt<=4) || ((t2->vert[j])->num_adjt<=4))
    return -1;
  
  /* do not generate vertices with to many adj's */
  if ((v1->num_adjv+v2->num_adjv)>(MAXNUMADJS+3))
    return -1;

#ifdef CAUTIOUS
  /* this means something has gone wrong (must be checked somehow) */
  if ((i==-1) || (j==-1) || (t1->vert[i]==t2->vert[j])) {
    printf("ERROR, something is serious wrong in the mesh\n");
    return -1;
  }
#endif
  
  /* remove the vert v2 from v1 and v2's comon adjv's */
  vertremovevert(t1->vert[i],v2,1);
  exchangeadjacencies(t1,i);
  vertremovevert(t2->vert[j],v2,2); 
  exchangeadjacencies(t2,j);

  /* remove the vertex cross-referencing */
  vertremovevert(v1,v2,3); 
  vertremovevert(v2,v1,4);

  /* The triangles should both be removed from all vertices */
  for(i=0; i<3; i++) {
    vertremovetri(t1->vert[i],t1);
    vertremovetri(t2->vert[i],t2);
  }

  /* calculate new vertex position */
  /* todo: this should be altered to use the vertice normals */
  v1->x=(v1->x+v2->x)/2;
  v1->y=(v1->y+v2->y)/2;
  v1->z=(v1->z+v2->z)/2;

  /* and the vertex v2's references moves to v1: */
  movevertrefs(v2,v1);

  /* update the altered triangles' data */
  for (i=0; i<v1->num_adjt; i++) {
    calctriedges(v1->adjt[i]);
    checktri(v1->adjt[i],edgesize);
  }

  /* check for, and prevent new loop problems */
  for (i=0; i<v1->num_adjv; i++)
    if (vertvertincommon(v1,v1->adjv[i])>2) {
      v1->donottouch=1;
      v1->adjv[i]->donottouch=1;
    }
  
  /* kill the vertices */
  killvert(v2);
  killtri(t1);
  killtri(t2);

  /* hooray the collapse worked' */
  return 0;
}


/* This is all initialization in order to prepare for the 
   mesh simplification */
void
initlists(float *v, int nv)
{
  int i,j,k;
  int poly,npolys,vertcount=0,polycount=0,tricount=0,count,degeneratecount = 0;
  int adjcount[4],adjnumber,adjtris;
  Vert *tmpvert,*vert,*vert0,*vert1;
  Tri *temptri, *tri, *nexttri, *tmptri;
  Edge *edge;

  /* the major lists */
  if (VERBOSE)
    printf("%s: Generating hash lists\n",MY_NAME);
  vertlist=makevertlist();
  trilist=maketrilist();

  /* use hashing to compute minimum number of vertices */
  npolys=nv/3;
  tmpvert=makevert();
  hashvertbegin();
  for(poly=0; poly<npolys; poly++) {
    tri = maketri();
    polycount++;
    inserttri(trilist,tri);
    for(i=0; i<3; i++) {
      tmpvert->x=v[poly*9+i*3];
      tmpvert->y=v[poly*9+i*3+1];
      tmpvert->z=v[poly*9+i*3+2];

      vert = hashvertadd(tmpvert);
      if(vert)
	/* the vertex exists */
	tri->vert[i] = vert;
      else  {
	/* add a new vertex to the list */
	tmpvert->index = vertcount;
	vertcount++;
	tri->vert[i] = tmpvert;
	insertvert(vertlist,tmpvert);
	tmpvert = makevert();
      }
    }
  }  
  hashvertend();
  
  /* removing degenerate triangles if any! */
  degeneratecount = 0;
  for (tri=trilist->first; tri;) {
    if ((tri->vert[0] == tri->vert[1]) ||
	(tri->vert[0] == tri->vert[2]) ||
	(tri->vert[1] == tri->vert[2])) {
      degeneratecount += 1;
      tmptri = tri->next;
      deletetri(trilist,tri);
      tri = tmptri;
    } else 
      tri = tri->next;
  }
  if (degeneratecount)
    printf("%d degenerate triangles eliminated\n",degeneratecount);
  
  printf("%s: Object consists of %d verts and %d triangles\n",MY_NAME,vertcount,polycount);  

  hashedgebegin(npolys);
  
  /* add all edges to hash table */
  tricount=0;
  for (tri=trilist->first; tri; tri=tri->next) {
    tricount++;
    hashedgeadd(tri,tri->vert[0],tri->vert[1]);
    hashedgeadd(tri,tri->vert[1],tri->vert[2]);
    hashedgeadd(tri,tri->vert[2],tri->vert[0]);
  }

  /* find the adjecant triangles for all edges */
  count = 0;
  for (tri=trilist->first; tri; tri=tri->next) {
    count += 1;
    for (i=0; i<3; i++) {
      if (tri->adj[i]) continue;
      vert0 = tri->vert[i];
      vert1 = tri->vert[suc[i]];
      for (edge=hashedgefind(vert0,vert1); edge; edge=edge->hnext) {
	nexttri = edge->tri;
	if(nexttri == tri) continue;
	for (j=0; j<3; j++) {
	  if (vert0 == nexttri->vert[j]) {
	    for (k=0; k<3; k++) {
	      if (k==j) continue;
	      if (vert1 == nexttri->vert[k]) {
		static t[4] = {0, 0, 2, 1};
		adjnumber = t[j+k];
		if (tri->adj[i]||nexttri->adj[adjnumber]) {		
		} else {			  
		  tri->adj[i] = nexttri;
		  nexttri->adj[adjnumber] = tri;
		}
	      }
	    }
	  }
	}
      }
    }
  }
  hashedgeend();
  if (VERBOSE)
    printf("%s: Computing adjacency information: done  \n",MY_NAME);

  /* compute edge adjacency statistics */
  for (i=0; i<4; i++)
    adjcount[i] = 0;
  for (tri=trilist->first; tri;) {
    count = (tri->adj[0] != 0) + (tri->adj[1] != 0) + (tri->adj[2] != 0);
    tri->adjcount = count;
    adjcount[count] += 1;
    if (count<3)
      for(i=0; i<3; i++) {
	if (! ((tri->vert[i])->edge))
	  (tri->vert[i])->edge=(tri->adj[i]!=0);
	if (! ((tri->vert[suc[i]])->edge))
	  (tri->vert[suc[i]])->edge=(tri->adj[i]!=0);
      }
    nexttri = tri->next;
    tri = nexttri;
  }

  /* update vertex adjacency */
  for (tri=trilist->first; tri;) {
    for (i=0; i<3; i++) {
      vertaddtri(tri->vert[i],tri);
      if (!isvertinvert(tri->vert[i],tri->vert[suc[i]]))
	vertaddvert(tri->vert[i],tri->vert[suc[i]]);
      if (!isvertinvert(tri->vert[i],tri->vert[suc[i+1]]))
	vertaddvert(tri->vert[i],tri->vert[suc[i+1]]);
    }
    tri = tri->next;
  } 

  /* compute triangle and vertex adjacency statistics and 
     normalize vertex coordinates */
  adjtris=count=0;
  vertcount=polycount=0;
  for (vert=vertlist->first; vert;) {
    count++;
    vert->x*=delta[0];
    vert->y*=delta[1];
    vert->z*=delta[2];
    vertcount+=vert->num_adjv;
    polycount+=vert->num_adjt;
    vert=vert->next;
  }

  /* print some statistics */
  if (VERBOSE) {
    printf("%s: adjacencies: 0:%d, 1:%d, 2:%d, 3:%d\n",MY_NAME,
	 adjcount[0],adjcount[1],adjcount[2],adjcount[3]);
    printf("%s: avg. adj.: %f tris, %f verts\n",MY_NAME,(float)polycount/count,
	 (float)vertcount/count);
  }

  if (VERBOSE)
    printf("%s: Caclulating surface normals\n",MY_NAME);
  /* calculate triangle normals and sidelengths*/
  count=0; edgelength=0.0;
  for (tri=trilist->first; tri;) {
    calc_tri_normal(tri);
    calctriedges(tri);
    for (i=0; i<3; i++)
      edgelength+=sqrt(tri->edgelength[i]);
    count++;
    tri=tri->next;
  }    
  edgelength=edgelength/3/count;
  if (VERBOSE) {
    printf("%s: Voxel dimensions: [%.4f;%.4f;%.4f]\n",MY_NAME,delta[0],delta[1],delta[2]);
    printf("%s: Mean triangle edgelength: %f \n",MY_NAME,edgelength);
  }
  count=0;
  for (vert=vertlist->first; vert;) {
    for (i=0; i<vert->num_adjv; i++)
      if (vertvertincommon(vert,vert->adjv[i])>2) {
	count+=!vert->donottouch+!(vert->adjv[i])->donottouch;
	vert->donottouch=1;
	(vert->adjv[i])->donottouch=1;
      }
    vert=vert->next;
  }    
  if (VERBOSE && count)
    printf("%s: %d vertices marked untouchable\n",MY_NAME,count);
}


/* Reduces the number of triangles in the mesh (using edgesize) and returns
   wether the new mesh has a triangle count under 'goal' triangles */
int
simplereduce(float edgesize,int oldvcount,int oldtcount, int goal) {
  int i,killedtris,count,vcount,tcount;
  int rcount;
  Tri *tri,*nexttri,*t;
  Vert *vert;

  edgesize*=edgesize;  
  removable=maketrilist();

  /* count vertices */
  vcount=0;
  for (vert=vertlist->first; vert;) {
    vcount++;
    vert=vert->next;
  }    
  /* count triangles */
  tcount=0;
  for (tri=trilist->first; tri;) {
    tcount++;    
    nexttri=tri->next;
    checktri(tri,edgesize);
    tri=nexttri;
  }    
  
  count=0;
  for (tri=removable->first; tri; ) {
    count++;

    for(i=0; i<3; i++) {
      if(tri->edgelength[i]<edgesize) {
	if (collapsetri(tri,tri->vert[i],tri->vert[suc[i]],edgesize)!=-1) {
	  /* two triangles and one vertex were removed */
	  vcount--;
	  tcount-=2;
	  break;
	} else {
	  /* the triangle could not be removed - something is wrong,
	     so it's removed from the removable list for now...*/
	  forcemovetri(tri);
	  break;
	}
      }
    }
    /* get the next triangle to remove */
    tri=removable->first;
  } 
  
  /* the removable list is empty now, free it */
  free(removable);
  if (tcount<goal)
    return 1;
  else 
    return 0;
}

  
/* 'main' routine */
Tcl_Obj 
*reduce(Tcl_Interp *interp, int offset, float xxm[2][3])
{
  int i,j,k,steps,vcount,tcount;
  float side,incr,sidelength;
  Tri *tri,*tri1,tri2;
  Vert *vert,*vert1;
  float *v=VERTICES;
  int nv=NUM_VERTICES;
  Tcl_Obj *resObj;

  initlists(VERTICES,NUM_VERTICES);
  free(VERTICES);

  vcount=0;
  for (vert=vertlist->first; vert;) {
    vcount++;
    vert=vert->next;
  }    
  tcount=0;
  for (tri=trilist->first; tri;) {
    tcount++;
    tri=tri->next;
  }    
  triseperate();

  tid4=clock();
  
  if ((REQPOLS<tcount)&&(REQPOLS!=0)) {
    printf("%s: Reducing to max. %d triangles\n",MY_NAME,REQPOLS);
    printf("\n  tris   verts  v's killed  ratio  max length\n");
    printf("-----------------------------------------------\n");
    sidelength=edgelength*15.0;
    steps=200;
    incr=sidelength/steps;
    
    for(side=incr; side<=sidelength; side+=incr)
      if (simplereduce(side,vcount,tcount,REQPOLS))
	break;
    printf("\n-----------------------------------------------\n\n");
  }
  
  tid5=clock();

  /* count vertices and update index nrs' */
  vcount=0;
  for (vert=vertlist->first; vert;) {
    vert->index=vcount++;
    vert=vert->next;
  }    

  /* count nr. of tris */
  tcount=0;
  for (tri=trilist->first; tri;) {
    tcount++;
    tri=tri->next;
  }    

  /* resObj = Tcl_NewListObj(0,NULL); */
  resObj = Tcl_NewListObj(0,NULL);
  resObj = dump(interp,vcount,tcount, offset, xxm);
  
  /* free verts & tris */
  for (tri=trilist->first; tri;) {
    tri1=tri->next;
    deletetri(trilist,tri);
    free(tri);
    tri=tri1;
  }    
  for (vert=vertlist->first; vert;) {
    vert1=vert->next;
    deletevert(vertlist,vert);
    free(vert);
    vert=vert1;
  }    
  free(trilist);
  free(vertlist);
  free(removable);

  return resObj;
}

float 
min(float a, float b)
{
  return a<b?a:b;
}
float 
max(float a, float b)
{
  return a>b?a:b;
}


Tcl_Obj 
*dump(Tcl_Interp *interp, int vcount, int tcount, int offset, float xxm[2][3])
{
  int i, j, k, n, s;
  double x, y, z;
  float sample, op1, ogx, ogy, ogz;
  float sum, mx, my, mz, dx, dy, dz, dst, ang;
  float xm, ym, zm, xx, yx, zx;
  Vert *vert;
  Tri *tri, *ttmp;
  Tcl_Obj *lvxt, *lnrm, *resObj;
  
  resObj = Tcl_NewListObj(0, NULL);
  sample = (float)SAMPLE;
  op1    = (float)(offset+1);

  ogx = (float)(XDIM-1)/2.;
  ogy = (float)(YDIM-1)/2.;
  ogz = (float)(ZDIM-1)/2.;
  
  xm = ym = zm = 99999999.;
  xx = yx = zx = -999999999.;

  fprintf(stderr, "\nNvert %d Ntria %d\n", vcount, tcount);
  fflush(stderr);
  lvxt = Tcl_NewListObj(0, NULL);
  lnrm = Tcl_NewListObj(0, NULL);
  for (tri=trilist->first; tri;) {    
    for (j=0; j<3; j++) {
      vert = tri->vert[j];

      xm = min(xm,vert->x);
      xx = max(xx,vert->x);
      ym = min(ym,vert->y);
      yx = max(yx,vert->y);
      zm = min(zm,vert->z);
      zx = max(zx,vert->z);

      x = (double)((vert->x + ogx - op1)/sample + xxm[0][0]);
      y = (double)((vert->y + ogy - op1)/sample + xxm[0][1]);
      z = (double)((vert->z + ogz - op1)/sample + xxm[0][2]);
      
      Tcl_ListObjAppendElement(interp, lvxt, Tcl_NewDoubleObj(x));
      Tcl_ListObjAppendElement(interp, lvxt, Tcl_NewDoubleObj(y));
      Tcl_ListObjAppendElement(interp, lvxt, Tcl_NewDoubleObj(z));

      mx = my = mz = 0.0;
      n=0;
      for(k=0;k<vert->num_adjt;k++) {
	ttmp = vert->adjt[k];

	mx += ttmp->nx;
	my += ttmp->ny;
	mz += ttmp->nz;
	
	dst = tri->nx*ttmp->nx + tri->ny*ttmp->ny + tri->nz*ttmp->nz;
	if (dst > 1.) dst = 1.0;
	if (dst < -1.) dst = -1.0;
	ang = acos(dst)*180./acos(-1.0);

	n++;
      }
      
      mx /= (float)n;
      my /= (float)n;
      mz /= (float)n;

      sum = sqrt(mx*mx + my*my + mz*mz);
      if (sum == 0.0) sum = 1.0;
      mx /= sum;
      my /= sum;
      mz /= sum;

      Tcl_ListObjAppendElement(interp, lnrm, Tcl_NewDoubleObj((double)mx));
      Tcl_ListObjAppendElement(interp, lnrm, Tcl_NewDoubleObj((double)my));
      Tcl_ListObjAppendElement(interp, lnrm, Tcl_NewDoubleObj((double)mz));
    }
    
    tri = tri->next;
  }
  
  Tcl_ListObjAppendElement(interp, resObj, lvxt);  
  Tcl_ListObjAppendElement(interp, resObj, lnrm);  

  fprintf(stderr, "\nBBOX VERTEX\n");
  fprintf(stderr, "%8.3f%8.3f%8.3f\n",xm,ym,zm);
  fprintf(stderr, "%8.3f%8.3f%8.3f\n",xx,yx,zx);

  return resObj;
}




