/* Collection of procefures and functions used
   for downsampling and filtering of 3D matrices 
   
   Jesper James Jensen 1995                        
   */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "polyr.h"
#include <tcl.h>

#define _FFT	       -1
#define _IFFT		1
#define Swap(a,b) {Temp=(a);(a)=(b);(b)=Temp;}

float *
downsample(float *data, int *size, int *newsize)
{
  float *newdata;                        /* New data matrix                     */
  float deltax,deltay,deltaz;            /* new sampling parameters             */
  int datasize;                          /* Size of new array < size of old     */
  int x,y,z;                             /* we need counters (!)                */
  float realx,realy,realz;               /* help variables                      */
  int xsize=size[0]; 
  int xysize=size[0]*size[1];            
  int newidx,idx;
  float yhelp, yp1help;
  float xfrac, yfrac, zfrac;
  float zhelp, zp1help, offset;          /* All theese are speed optimizations  */
  
  /* allocate memory for new array */
  datasize=newsize[0]*newsize[1]*newsize[2];
  if(!(newdata=(float *)ckalloc(datasize*sizeof(float)))) {
    fprintf(stderr,"Memory allocation problems (downsample)\n");
    exit(1);
  }
  
  /* ok, new sampling parameters */
  deltax=((float)size[0])/(newsize[0]);
  deltay=((float)size[1])/(newsize[1]);
  deltaz=((float)size[2])/(newsize[2]);
  
  /* first some defines to nicen' the code a bit: */
#define IPOLDATA(off,delta)     ((data[idx]*((1.-delta)) + data[idx+xsize]*(delta)))
#define IPOL(data1,data2,delta) (data1*(1.-(delta)) + data2*(delta))
  /* It's just a simple case of tri-linear interpolation through the whole array */
  for(realz=0, z=0; z<newsize[2]; z++, realz+=deltaz) {
    zp1help=xysize+(zhelp=floor(realz)*xysize);
    zfrac=realz-floor(realz);
    for(realy=0, y=0; y<newsize[1]; y++, realy+=deltay) {
      yp1help=xsize+(yhelp=floor(realy)*xsize);
      yfrac=realy-floor(realy);
      newidx=y*newsize[0]+z*newsize[0]*newsize[1]; 
      for(realx=0, x=0; x<newsize[0]; x++, realx+=deltax) {
	idx=yhelp+zhelp+floor(realx);
	newdata[x+newidx]=
	  IPOL(
	       IPOL(
		    IPOLDATA(idx,yfrac),
		    IPOLDATA(idx+1,yfrac),
		    xfrac=(realx-floor(realx))),
	       IPOL(		  
		    IPOLDATA(idx+xysize,yfrac),
		    IPOLDATA(idx+xysize+1,yfrac),
		    xfrac),
	       zfrac);
      }
    }
  }
  /* return the downsampled array */
  return newdata;
}


void
expandmatrix(float **data, int *size, int *newsize)
{
  /* expands the matrix data to newsize>=size */
  float *newdata;
  int x,y,z;
  int xdim=size[0],xydim=size[0]*size[1];
  int newxdim=newsize[0],newxydim=newsize[0]*newsize[1];

  /* allocate memory for new array */
  if(!(newdata=(float *)ckalloc(2*newsize[0]*newsize[1]*newsize[2]*sizeof(float)))) {
    fprintf(stderr,"Memory allocation problems (exapandmatrix)\n");
    exit(1);
  }

  /* copy data in to a complex matrix */
  for(x=0; x<size[0]; x++)
    for(y=0; y<size[0]; y++)
      for(z=0; z<size[0]; z++) 
	newdata[2*(x+y*newxdim+z*newxydim)]=(*data)[x+y*xdim+z*xydim];

  /* kill old matrix, and copy new one */
  ckfree((char *) *data);
  (*data)=newdata;
}


void
truncmatrix(float **data, int *size, int *newsize)
{
  /* truncs the matrix data to newsize<=size */
  float *newdata;
  int x,y,z;
  int xdim=size[0],xydim=size[0]*size[1];
  int newxdim=newsize[0],newxydim=newsize[0]*newsize[1];

  /* allocate memory for new array */
  if(!(newdata=(float *)ckalloc(newsize[0]*newsize[1]*newsize[2]*sizeof(float)))) {
    fprintf(stderr,"Memory allocation problems (truncmatrix)\n");
    exit(1);
  }

  /* copy the real data */
  for(x=0; x<newsize[0]; x++)
    for(y=0; y<newsize[0]; y++)
      for(z=0; z<newsize[0]; z++) 
	newdata[x+y*newxdim+z*newxydim]=(*data)[2*(x+y*xdim+z*xydim)];

  /* kill old matrix, and copy new one */
  ckfree((char *) *data);
  (*data)=newdata;
}


int 
NextRadix2(int i)
{
  /* This was stolen from Hrr. Peter Toft's '3dtools.c' */
  return (int)floor(0.5+exp(log(2.)*ceil(log(i)/log(2.))));
}


void 
ComplexNdimFFT(float *data, int *nn, int ndim, int isign)
{
  /* slightly modified version of JJJ & PAP's version of
     nummerical recepies ndimfft() */  
  int idim,count;
  unsigned long i1,i2,i3,i2rev,i3rev,ip1,ip2,ip3,ifp1,ifp2;
  unsigned long ibit,k1,k2,n,nprev,nrem,ntot;
  float tempi,tempr,Temp;
  double theta,wi,wpi,wpr,wr,wtemp;

  for(ntot=1,idim=1;idim<=ndim;idim++)
     ntot *= nn[idim-1];
  nprev=1;
  for(idim=ndim;idim>=1;idim--) {
     n=nn[idim-1];
     nrem=ntot/(n*nprev);
     ip1=nprev<<1;
     ip2=ip1*n;
     ip3=ip2*nrem;
     i2rev=1;
     for(i2=1;i2<=ip2;i2+=ip1) {
        if(i2<i2rev) {
          for(i1=i2;i1<=i2+ip1-2;i1+=2) {
             for(i3=i1;i3<=ip3;i3+=ip2) {
                i3rev=i2rev+i3-i2;
                Swap(data[i3],data[i3rev]);
                Swap(data[i3+1],data[i3rev+1]);
	     }
     	  }
	}
        ibit=ip2>>1;
        while(ibit>=ip1 && i2rev>ibit) {
             i2rev-=ibit;
             ibit>>=1;   
	}
        i2rev+=ibit;
     }
     ifp1=ip1;
     while(ifp1<ip2) {
  	ifp2=ifp1<<1;
	theta=isign*6.28318530717959/(ifp2/ip1);
	wpr=cos(theta);
	wpi=sin(theta);
	wr=1.0;
	wi=0.0;
	for(i3=1;i3<=ifp1;i3+=ip1) {
	   for(i1=i3;i1<=i3+ip1-2;i1+=2) {
              for(i2=i1;i2<=ip3;i2+=ifp2) {
		 k1=i2;
		 k2=k1+ifp1;
		 tempr=(float)wr*data[k2]-(float)wi*data[k2+1];
                 tempi=(float)wr*data[k2+1]+(float)wi*data[k2];
		 data[k2]=data[k1]-tempr;
		 data[k2+1]=data[k1+1]-tempi;
		 data[k1] += tempr;
		 data[k1+1] += tempi;
       	      }
	   }
           wr=(wtemp=wr)*wpr-wi*wpi;
           wi=wi*wpr+wtemp*wpi;
	}
      	ifp1=ifp2;
    }
    nprev*=n;
  }
  if (isign==_IFFT) {
    for(count=1; count<=ntot*2; count++) {
      data[count]/=ntot;
    }
  }
}


void
fourierfilter(float *data, int *size, int *newsize)
{
  int fftsize[3],cutoff[3];
  int x,y,z,xsize,xysize,offset;

  /* Find a size we can Fourier transform */
  fftsize[0]=NextRadix2(size[0]);
  fftsize[1]=NextRadix2(size[1]);
  fftsize[2]=NextRadix2(size[2]);

  /* expand and FFT */
  expandmatrix(&data,size,fftsize);
  ComplexNdimFFT(&data[-1], fftsize, 3, _FFT); 

  /* it is an open question if the exact limit frequenzies should
     be removed or not. Adjust it to your liking with different
     rounding in the calculation of cutoff[]! */
  cutoff[0]=(fftsize[0]/2-1)*(1-((float)newsize[0]/size[0]));
  cutoff[1]=(fftsize[1]/2-1)*(1-((float)newsize[1]/size[1]));
  cutoff[2]=(fftsize[2]/2-1)*(1-((float)newsize[2]/size[2]));
  
  /* ok - now we remove the to high frequenzies 
     with a simple lowpass square filter */
  xsize=fftsize[0]; xysize=fftsize[0]*fftsize[1];
  for(y=fftsize[1]/2-cutoff[1]; y<=fftsize[1]/2+cutoff[1]; y++)
    for(z=fftsize[2]/2-cutoff[2]; z<=fftsize[2]/2+cutoff[2]; z++) 
      offset=y*xsize+z*xysize; 
      for(x=fftsize[0]/2-cutoff[0]; x<=fftsize[0]/2+cutoff[0]; x++) {
	data[2*(x+offset)]=0;
	data[2*(x+offset)+1]=0;
      }

  /* IFFT and truncate, now the matrix is filtered */
  ComplexNdimFFT(&data[-1], fftsize, 3, _IFFT);
  truncmatrix(&data,fftsize,size);
}


/* mean filters the data array with a boxfilter of size 'boxsize' */
void
boxfilter(float **data, int *size, int *boxsize) {
  float *newdata;
  int i,j,k,x,y,z,count;
  int datap,dataphelp,newdatap;
  int filterx,filtery,filterz;
  int imin,imax,jmin,jmax,kmin,kmax;
  int xsize=size[0], xysize=size[0]*size[1];
  float sum, volume;

#define max(a,b) ((a)>(b)) ? (a) : (b)
#define min(a,b) ((a)>(b)) ? (b) : (a)
  /* boxsize must be odd */
  if (((boxsize[0]-1)%2)||((boxsize[2]-1)%2)||((boxsize[2]-1)%2)) {
    fprintf(stderr,"%s: ERROR in filter dimensions\n",MY_NAME);
    exit(1);
  }
  filterx=(boxsize[0]-1)/2;
  filtery=(boxsize[1]-1)/2;
  filterz=(boxsize[2]-1)/2;  

  if(!(newdata=(float *)ckalloc(size[0]*size[1]*size[2]*sizeof(float)))) {
    fprintf(stderr,"Memory allocation problems (downsample)\n");
    exit(1);
  }
  /* the filter loop */
  for(j=0;j<size[1]; j++) {
    if (VERBOSE) {
      fprintf(stderr,"%s: Filtering: %3.0f%%\r",MY_NAME,((float)j+1)/size[1]*100);
      fflush(stdout);
    }
    jmin=max(j-filtery,0);
    jmax=min(j+filtery+1,size[1]);
    for(k=0;k<size[2]; k++) {
      newdatap=j*xsize+k*xysize;
      kmin=max(k-filterz,0);
      kmax=min(k+filterz+1,size[2]);
      for(i=0;i<size[0];i++) {
	imin=max(i-filterx,0);
	imax=min(i+filterx+1,size[0]);
	count=0;
	sum=0.0;
	for(z=kmin; z<kmax; z++) {
	  dataphelp=z*xysize+imin;
	  for(y=jmin; y<jmax; y++) {
	    datap=y*xsize+dataphelp;
	    for(x=imin; x<imax; x++) {
	      sum+=(*data)[datap];
	      datap++;
	      count++;
	    }
	  }
	}
	newdata[newdatap]=sum/count; 
	newdatap++;
      }
    }
  }
  if (VERBOSE)
    fprintf(stderr,"%s: Filtering done   \n",MY_NAME);
  ckfree((char *)*data);
  (*data)=newdata;
}


void
printmatrix(float *data, int *size)
{
  int x,y,z;

  for(z=0; z<size[2]; z++) {
    fprintf(stderr,"\n");
    for(y=0; y<size[1]; y++) {
      fprintf(stderr,"\n");
      for(x=0; x<size[0]; x++)
	printf("%8.2f",data[(int)x+y*size[0]+z*size[0]*size[1]]);
    }
  }
  fprintf(stderr,"\n");
}



/* a more readable (and much slower!) version of the downsample routine */
/*
#define IPOLY(x,y,z,dy) ( (data[(int)((x)+floor(y)*xsize+(z)*xysize)]*(1-(dy)) \
			   + data[(int)((x)+floor(y+1)*xsize+(z)*xysize)]*(dy)) )
#define IPOL(data1,data2,delta) ( data1*(1-(delta)) + data2*(delta) )
    
    for(realx=0, x=0; x<newsize[0]; x++, realx+=deltax) {
      for(realy=0, y=0; y<newsize[1]; y++, realy+=deltay) {
	for(realz=0, z=0; z<newsize[2]; z++, realz+=deltaz) {
	  newdata[x+y*newsize[0]+z*newsize[0]*newsize[1]]=
	  IPOL(
	       IPOL(
		    IPOLY(floor(realx),realy,floor(realz),realy-floor(realy)),
		    IPOLY(floor(realx+1),realy,floor(realz),realy-floor(realy)),
		    realx-floor(realx)),
	       IPOL(		  
		    IPOLY(floor(realx),realy,floor(realz+1),realy-floor(realy)),
		    IPOLY(floor(realx+1),realy,floor(realz+1),realy-floor(realy)),
		    realx-floor(realx)),
	       realz-floor(realz)
	       );
	}
      }
    }
*/



