//------------------------------------------------
// MarchingCubes
//------------------------------------------------
//
// MarchingCubes Command Line interface
// Version 0.2 - 12/08/2002
//
// Thomas Lewiner thomas.lewiner@polytechnique.org
// Math Dept, PUC-Rio
//
// Translated to C by Ziad S. Saad November 30/04
//________________________________________________


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <tcl.h>
#include "MarchingCubes.h"

#ifndef DLLEXPORT
#define DLLEXPORT __declspec(dllexport)
#endif 

int
MCTclObjCmd(ClientData clientData,
	    Tcl_Interp *interp,
	    int objc,
	    Tcl_Obj *const objv[]);

static char Obj_Types[][10] = { {"Cushin"}, {"Sphere"}, {"Plane"}, {"Cassini"}, {"Blooby"}, {"Chair"}, {"Cyclide"}, {"2 Torus"}, {"mc case"}, {"Drip"} };
void mc_help(void)
{
  int i;
  printf(  "Creates an isosurface for a predetermined set of objects\n"
	   "or a 1D volume of choice. Isosurface is the envelope of\n"
	   "volumetric region = 0.\n"
	   "If region is spherical of radius 5, isosurface would be\n"
	   " x^2+y^2+z^2-r^2 = 0\n"
	   "Usage 1: mc <obj_type>\n"
	   "           obj_type is a number between 0 and 9\n");
  for (i=0; i<10;++i) {
    printf(  "           %d: %s\n", i, Obj_Types[i]);   
  }
  printf(  "Usage 2: mc <Vol.1D> <Res> \n"
	   "           Vol.1D is the (Res x Res x Res) volume in 1D format.\n"
	   "           You can't use a binary file where you have 0 and 1,\n"
	   "           like in a segmented volume, you need to create a gradient\n"
	   "           over zero. \n"
	   " Example 1: If you have a mask with 0 outside \n"
	   "           the brain and 1 inside, turn all 0s into -1.\n"
	   "           Say Anat is a 128x128x128 volume for example:\n"
	   "           3dAutomask -prefix Mask Anat+orig.            \n"
	   "           3dcalc -datum float -a Mask+orig. -expr '( -1*(1-bool(a))+bool(a) )' -prefix GradMask\n"
	   "           3dmaskdump GradMask+orig. > GradMask.1D\n"
	   "           mc GradMask.1D 128\n"
	   "           quickspec -tn 1D testNodes.1D testFaces.1D\n"
	   "           suma -spec quick.spec\n"
	   "\n"
	   "Usage 3: mc <Vol.1D> <ResX ResY ResZ> \n"
	   "         Same as usage 2 but allowing for non cubical volumes.\n"
	   "         in the 1D file, the X dimension changes first, followed\n"
	   "         by Y followed by Z. \n"
	   "         In other terms: 1Drow = X + Nx Y + NxNy Z\n"
	   " Example 2: You have an ROI data set with non zero values where\n"
	   "            forming volumes of interest. To extract isosurface of \n"
	   "            value 3 from your ROI dataset COPY_closeup+orig:\n"
	   "         rm Mask* GradMask*\n"
	   "         set vol = COPY_closeup+orig\n"
	   "         set sz_st = `3dAttribute -name DATASET_DIMENSIONS $vol`\n"
	   "         set sz = ($sz_st[3] $sz_st[4] $sz_st[5])\n"
	   "         3dcalc -datum float -a $vol -expr '( -1*(1-equals(a,3))+equals(a,3) )' -prefix GradMask\n"
	   "         3dmaskdump GradMask+orig. > GradMask.1D\n"
	   "         mc GradMask.1D $sz[1] $sz[2] $sz[3]\n"
	   "         quickspec -tn 1D testNodes.1D testFaces.1D\n"
	   "         suma -spec quick.spec \n"
	   "\n" );
            
}
/* usage:
   mc 
*/
//_____________________________________________________________________________
// Init funcion for tcl
int DLLEXPORT Mctcl_Init (Tcl_Interp *interp) 
{
  if (Tcl_InitStubs(interp, "8.0" , 0) == NULL) {
    return TCL_ERROR;
  }

  Tcl_CreateObjCommand(interp, "mctcl", MCTclObjCmd, (ClientData) NULL, (Tcl_CmdDeleteProc *)NULL);

  Tcl_PkgProvide(interp,"mctcl","0.1");

  return TCL_OK;
}

//_____________________________________________________________________________
// main tcl function called by mctcl
int
MCTclObjCmd(ClientData clientData,
	    Tcl_Interp *interp,
	    int objc,
	    Tcl_Obj *const objv[]) 
     //-----------------------------------------------------------------------------
{
  MCB *mcp ;
  int sizes, nelt, Resx, Resy, Resz, i, j, k;  
  int v1, v2, v3; 
  float nx,ny,nz, x1,y1,z1,x2,y2,z2,x3,y3,z3;
  float norm, ax, ay, az, bx, by, bz;
  double val;
  Tcl_Obj **OMap, **OSizes, **OPts, *allObj, *trgObj, *nrmObj;
  Tcl_Obj *vec1Obj, *vec2Obj, *vec3Obj;

  if (objc < 2) {
    Tcl_WrongNumArgs(interp, 1, objv, "<value-list>");
    return TCL_ERROR;
  }
  
  /* Init general table */
  mcp = MarchingCubes(-1, -1, -1);

  /* read data */
  Tcl_ListObjGetElements(interp,objv[1],&nelt,&OMap);
  Tcl_ListObjGetElements(interp,OMap[0],&sizes,&OSizes);
  Tcl_ListObjGetElements(interp,OMap[1],&sizes,&OPts);
  Tcl_GetIntFromObj(interp, OSizes[0], &Resx);
  Tcl_GetIntFromObj(interp, OSizes[1], &Resy);
  Tcl_GetIntFromObj(interp, OSizes[2], &Resz);
  set_resolution( mcp, Resx, Resy, Resz) ;
  init_all(mcp) ;

  if (1) {
    // Data de test , voir MarchingCube.c , 
    // routine compute_data
    compute_data(*mcp, 1);
  } else {
    for (k=0; k<Resz; k++) {
      for (j=0; j<Resy; j++) {
	for (i=0; i<Resx; i++) {
	  /* BEWARE ! X fastest ! */
	  Tcl_GetDoubleFromObj(interp, OPts[i+j*Resx+k*Resx*Resy],&val);
	  set_data( mcp, val, i, j, k);
	}
      }
    }
    fprintf(stderr, "idx %d\n",i+j*Resx+k*Resy*Resx);
  }

  run(mcp);
  
  allObj = Tcl_NewListObj(0, NULL);
  fprintf(stderr, "Nb vertices : %d\n",mcp->nverts);
  fprintf(stderr, "Nb triangle : %d\n",mcp->ntrigs);
  for ( i=0; i < mcp->ntrigs; i++ )
    {
      trgObj = Tcl_NewListObj(0, NULL);
      v1 = mcp->triangles[i].v1;
      v2 = mcp->triangles[i].v2;
      v3 = mcp->triangles[i].v3;
      
      nrmObj  = Tcl_NewListObj(0, NULL);
      vec1Obj = Tcl_NewListObj(0, NULL);
      vec2Obj = Tcl_NewListObj(0, NULL);
      vec3Obj = Tcl_NewListObj(0, NULL);
      x1 = mcp->vertices[v1].x;
      y1 = mcp->vertices[v1].y;
      z1 = mcp->vertices[v1].z;
      
      x2 = mcp->vertices[v2].x;
      y2 = mcp->vertices[v2].y;
      z2 = mcp->vertices[v2].z;
      
      x3 = mcp->vertices[v3].x;
      y3 = mcp->vertices[v3].y;
      z3 = mcp->vertices[v3].z;
      
      /* remplit vecteur */
      /* 1er point du triangle */
      Tcl_ListObjAppendElement(interp, vec1Obj, Tcl_NewDoubleObj((double)(x1)));
      Tcl_ListObjAppendElement(interp, vec1Obj, Tcl_NewDoubleObj((double)(y1)));
      Tcl_ListObjAppendElement(interp, vec1Obj, Tcl_NewDoubleObj((double)(z1)));
      
      Tcl_ListObjAppendElement(interp, vec2Obj, Tcl_NewDoubleObj((double)(x2)));
      Tcl_ListObjAppendElement(interp, vec2Obj, Tcl_NewDoubleObj((double)(y2)));
      Tcl_ListObjAppendElement(interp, vec2Obj, Tcl_NewDoubleObj((double)(z2)));
      
      Tcl_ListObjAppendElement(interp, vec3Obj, Tcl_NewDoubleObj((double)(x3)));
      Tcl_ListObjAppendElement(interp, vec3Obj, Tcl_NewDoubleObj((double)(y3)));
      Tcl_ListObjAppendElement(interp, vec3Obj, Tcl_NewDoubleObj((double)(z3)));
      
      ax = x1 - x3; ay = y1 - y3; az = z1 - z3;
      bx = x2 - x3; by = y2 - y3; bz = z2 - z3;

      nx = ay * bz - az * by;
      ny = az * bx - ax * bz;
      nz = ax * by - ay * bx;
      
      norm = sqrt(nx*nx + ny*ny + nz*nz);
      if (norm < 0.0001) norm=1.;
      nx /= norm;
      ny /= norm;
      nz /= norm;
      
      fprintf(stderr, "\nTriangle %d\n", i);
      fprintf(stderr, "v1 %d v2 %d v3 %d\n",v1,v2,v3);
      fprintf(stderr, "vec1 %6.3f %6.3f %6.3f\n",x1,y1,z1);
      fprintf(stderr, "vec2 %6.3f %6.3f %6.3f\n",x2,y2,z2);
      fprintf(stderr, "vec3 %6.3f %6.3f %6.3f\n",x3,y3,z3);
      fprintf(stderr, "norm %6.3f %6.3f %6.3f\n",nx,ny,nz);

      /* remplit normale */
      Tcl_ListObjAppendElement(interp, nrmObj, Tcl_NewDoubleObj((double)(nx)));
      Tcl_ListObjAppendElement(interp, nrmObj, Tcl_NewDoubleObj((double)(ny)));
      Tcl_ListObjAppendElement(interp, nrmObj, Tcl_NewDoubleObj((double)(nz)));
      
      /* Met normale - vecteur dans triangle */
      Tcl_ListObjAppendElement(interp, trgObj, vec1Obj);
      Tcl_ListObjAppendElement(interp, trgObj, nrmObj);
      Tcl_ListObjAppendElement(interp, trgObj, vec2Obj);
      Tcl_ListObjAppendElement(interp, trgObj, nrmObj);
      Tcl_ListObjAppendElement(interp, trgObj, vec3Obj);
      Tcl_ListObjAppendElement(interp, trgObj, nrmObj);
      /* append le triangle */
      Tcl_ListObjAppendElement(interp, allObj, trgObj);
    }
  Tcl_SetObjResult(interp, allObj);
  clean_temps(mcp);

#if 0
  writePLY(mcp, "test.ply", false) ;
  write1Dmcb(mcp);
#endif

  clean_all(mcp) ;
  free(mcp);

  return TCL_OK;
}




